/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DTOUR_H
#define DTOUR_H

#include <QWidget>
#include <QPainterPath>

/**
 * @brief The DTour class provides an interactive guided tour of the application.
 *
 * This widget acts as an overlay on top of the parent window. It guides the user
 * through a sequence of steps by dimming the background and highlighting specific
 * areas (widgets) of the interface to explain their functionality.
 */
class DTour : public QWidget
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the DTour widget.
   * @param parent The parent widget over which the tour will be displayed.
   * @param sizes A map containing geometry data or dimensions used to calculate highlight zones.
   */
  DTour(QWidget *parent, QMap<QString, uint> sizes);

  /**
   * @brief Repositions the tour overlay.
   *
   * Ensures the tour widget stays synchronized with the parent widget's position
   * if the main window is moved.
   */
  void moveWindow();

private:
  /**
   * @brief Configures the graphical elements for the first step of the tour.
   */
  void drawStep1();

  /**
   * @brief Configures the graphical elements for the second step of the tour.
   */
  void drawStep2();

  /**
   * @brief Configures the graphical elements for the third step of the tour.
   */
  void drawStep3();

  /**
   * @brief Configures the graphical elements for the fourth step of the tour.
   */
  void drawStep4();

  /**
   * @brief Configures the graphical elements for the final step of the tour.
   */
  void drawLastStep();

  unsigned int step;
  QWidget *parent;
  QMap<QString, uint> sizes;
  QPolygon polygon;
  QBrush highlightedBrush;
  QBrush grayBrush;
  QPainterPath painterPath;
  QFont font;
//  QPainter painter;
  unsigned int windowsValor;

protected:
  /**
   * @brief Handles the painting event.
   *
   * Draws the semi-transparent background and the clear "hole" (highlight)
   * over the current step's target area, along with explanatory text.
   *
   * @param event The paint event parameters.
   */
  void paintEvent(QPaintEvent* event);

  /**
   * @brief Handles mouse press events.
   *
   * Detects clicks to advance the tour to the next step.
   *
   * @param event The mouse event parameters.
   */
  void mousePressEvent(QMouseEvent *event);

  /**
   * @brief Handles move events.
   *
   * Used to track movement and keep the overlay aligned.
   *
   * @param event The move event parameters.
   */
  void moveEvent(QMoveEvent *event);

};

#endif // DTOUR_H
