/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DTABLEWIDGET_H
#define DTABLEWIDGET_H

#include <QTableWidget>

#include "dbms.h"

/**
 * @brief The DTableWidget class extends QTableWidget with custom clipboard and database features.
 *
 * This widget provides enhanced functionality for displaying tabular data, including
 * a custom context menu for Copy/Paste operations and signals to execute SQL statements
 * generated from pasted data.
 */
class DTableWidget : public QTableWidget
{
  Q_OBJECT

//  Q_PROPERTY(QString tableName READ getTableName WRITE setTableName CONSTANT)

public:
  /**
   * @brief Constructs the DTableWidget.
   * @param headers A list of strings to set as the horizontal header labels.
   * @param selectionMode The item selection mode (default is SingleSelection).
   */
  DTableWidget(QStringList headers = QStringList(), QAbstractItemView::SelectionMode selectionMode = QAbstractItemView::SingleSelection);

  /**
   * @brief Sets the horizontal header labels.
   * @param headers The list of strings to display in the header.
   */
  void setHeaders(QStringList headers);

  /**
   * @brief Populates the table with data.
   * @param rows Pointer to a list of rows, where each row is a QStringList of cell values.
   */
  void fillTable(QList<QStringList> *rows);

  /**
   * @brief Retranslates the user interface.
   *
   * Updates context menu actions (Copy/Paste) to reflect the current language.
   */
  void retranslateUI();

  /**
   * @brief Enables or disables the Paste action in the context menu.
   * @param enabled True to enable pasting, false to disable it.
   */
  void setPasteActionEnabled(bool enabled = true);

  /**
   * @brief Sets the name of the database table associated with this widget.
   * @param tableName The name of the table.
   */
  void setTableName(QString tableName);

  /**
   * @brief Retrieves the name of the associated database table.
   * @return A QString containing the table name.
   */
  QString getTableName();

private:
  QMenu *mainMenu;
  QAction *copyAction;
  QAction *pasteAction;
  QString tableName;
  QStringList statementsToExecute;

signals:
  /**
   * @brief Emitted when a paste operation generates SQL statements or commands.
   * @param statements A list of strings representing the commands to be executed.
   */
  void executeStatements(QStringList statements);

public slots:
  /**
   * @brief Slot to handle cell click events.
   */
  void cellClickedSlot();

  /**
   * @brief Copies the selected cells to the system clipboard.
   */
  void copy();

  /**
   * @brief Pastes data from the system clipboard into the table.
   *
   * This may trigger the generation of executeStatements depending on the implementation.
   */
  void paste();

protected:
  /**
   * @brief Handles the context menu event.
   *
   * Shows the custom menu with Copy/Paste actions when the user right-clicks.
   * @param event The context menu event parameters.
   */
  void contextMenuEvent(QContextMenuEvent *event);
};

#endif // DTABLEWIDGET_H
