/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DTABLEVIEW_H
#define DTABLEVIEW_H

#include <QTableView>
#include <QHeaderView>

class QStandardItemModel;
class QStandardItem;
class DItemDelegate;

/**
 * @brief A customized horizontal header view.
 *
 * This class extends QHeaderView to provide specific default behaviors,
 * such as enabling clickable sections for sorting or selection.
 */
class DHeaderView : public QHeaderView
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the DHeaderView.
   *
   * Initializes the header as Horizontal and enables clickable sections.
   */
  DHeaderView():QHeaderView(Qt::Horizontal) {
    setSectionsClickable(true);
    //connect(this, SIGNAL(sectionClicked(int)), this, SLOT(sectionClicked(int)));
  }

//signals:
  //void clickedSection(int index);

//public slots:
  //void sectionClicked(int index);
};

/**
 * @brief The DTableView class provides a generic, data-driven table widget.
 *
 * Inheriting from QTableView, this class manages its own QStandardItemModel
 * and facilitates populating data from string lists. It supports custom headers,
 * built-in sorting logic, loading progress signals, and configurable read-only states.
 */
class DTableView : public QTableView
{
  Q_OBJECT
//  Q_PROPERTY(bool readOnly READ getReadOnly WRITE setReadOnly)

public:
  /**
   * @brief Constructs the DTableView.
   * @param headers Pointer to a list of header configurations.
   * @param selectionMode The item selection mode (default is SingleSelection).
   */
  DTableView(QList<QStringList> *headers = new QList<QStringList>(), QAbstractItemView::SelectionMode selectionMode = QAbstractItemView::SingleSelection);

  /**
   * @brief Sets the read-only state of the table.
   * @param readOnly True to prevent editing, false to allow it.
   */
  void setReadOnly(bool readOnly);

  /**
   * @brief Checks if the table is in read-only mode.
   * @return True if read-only, false otherwise.
   */
  bool getReadOnly();

  /**
   * @brief Populates the internal model with data.
   *
   * @param modelData Pointer to a list of rows (each row is a QStringList).
   * @param readOnly Whether the items should be editable.
   * @param orderColumn The column index to use for initial sorting.
   * @param modelDataHasIcons If true, specific data fields are interpreted as icon paths.
   */
  void setModelData(QList<QStringList> *modelData, bool readOnly = true, unsigned int orderColumn = 0, bool modelDataHasIcons = true);

  /**
   * @brief Retrieves data from a specific model index.
   * @param index The model index.
   * @param role The data role (e.g., Qt::DisplayRole).
   * @param column Optional column override.
   * @return The data as a QVariant.
   */
  QVariant indexData(const QModelIndex &index, int role = Qt::DisplayRole, int column = -1);

  /**
   * @brief Retrieves data from specific row and column coordinates.
   * @param row The row index.
   * @param column The column index.
   * @param role The data role.
   * @return The data as a QVariant.
   */
  QVariant indexData(const unsigned int row, const unsigned int column, int role = Qt::DisplayRole);

  /**
   * @brief Configures the table headers using a structured list.
   *
   * The inner QStringList typically contains metadata like [Name, DelegateType, Icon, Alignment].
   * @param headers Pointer to the list of header definitions.
   */
  void setHeaders(QList<QStringList> *headers);

  /**
   * @brief Configures the table headers using a simple list of names.
   * @param headers List of column names.
   */
  void setHeaders(QStringList headers);

  /**
   * @brief Refreshes the headers based on the internal configuration.
   */
  void setHeaders();

  /**
   * @brief Retrieves the current header configuration.
   * @return Pointer to the list of header definitions.
   */
  QList<QStringList> *getHeaders();

  /**
   * @brief Retrieves the index of the currently selected item.
   * @return The QModelIndex of the current item.
   */
  QModelIndex currentIndexItem();

  /**
   * @brief Selects an item in the table based on text content.
   * @param item The text to search for.
   * @param column The column index to search within.
   */
  void setCurrentItem(QString item, unsigned int column);

  /**
   * @brief Retrieves the currently selected standard item.
   * @return Pointer to the QStandardItem.
   */
  QStandardItem *currentItem();

  /**
   * @brief Retrieves all items in a specific row.
   * @param row The row index.
   * @return A list of pointers to QStandardItem objects in that row.
   */
  QList<QStandardItem *> getRow(int row);

  int sortingColumn;
  Qt::SortOrder sortOrder;
  QBrush originalBackground;

  /**
   * @brief Gets the total number of rows in the model.
   * @return The row count.
   */
  unsigned int rowCount();

  QStandardItemModel *itemModel;

private:
  QList<QStringList> *headersList; //tablesDTableViewHeaders->append(QStringList() << tr("Version") << NoDelegate << "" << "Right");
  bool readOnly;
  QMenu *mainMenu;
  DHeaderView *dHeaderView;

public slots:
  /**
   * @brief Adds a new empty row to the table.
   */
  void addRow();

  /**
   * @brief Duplicates an existing row.
   * @param line The index of the row to clone.
   */
  void cloneLine(unsigned int line);

private slots:
  /**
   * @brief Slot called when an item's data changes.
   * @param item The item that changed.
   */
  void itemChangedSlot(QStandardItem *item);
//  void doubleClickedSlot(const QModelIndex &index);

  /**
   * @brief Slot called when a header section is clicked (for sorting).
   * @param index The index of the column clicked.
   */
  void sortingColumnSlot(int index);
//  void clickedSlot(const QModelIndex &index);

signals:
  /**
   * @brief Signal emitted when a data loading process begins.
   * @param message Description of the task.
   * @param timeout Expected timeout.
   * @param progress Initial progress value.
   */
  void loadStarted(QString message, unsigned int timeout, double progress);

  /**
   * @brief Signal emitted when a data loading process finishes.
   * @param message Completion message.
   * @param timeout Duration to show the completion message.
   * @param progress Final progress value.
   */
  void loadFinished(QString message, unsigned int timeout, double progress);

  /**
   * @brief Signal emitted to update loading progress.
   * @param message Current status message.
   * @param timeout Duration for the message.
   * @param progress Current progress value.
   */
  void loadProgress(QString message, unsigned int timeout, double progress);

  /**
   * @brief Signal emitted when a specific item has been modified.
   * @param item The modified QStandardItem.
   */
  void itemChanged(QStandardItem *item);
//  void rowDoubleClicked(QList<QVariant> row);
//  void rowClicked(QList<QVariant> row);

//protected:
//  void contextMenuEvent(QContextMenuEvent *event);
};

#endif // DTABLEVIEW_H
