/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DSOUNDEFFECT_H
#define DSOUNDEFFECT_H

#include <QObject>

/**
 * @brief The DSoundEffect class provides static access to application sound effects.
 *
 * This utility class manages audio feedback for the user interface. It contains
 * static methods to trigger specific sounds for events such as errors,
 * query execution, window operations, and screenshots.
 */
class DSoundEffect : public QObject
{
  Q_OBJECT

public:
  /**
   * @brief Plays the alarm sound effect.
   *
   * Typically used for alerts or time-sensitive notifications.
   */
  static void Alarm();

  /**
   * @brief Plays the sound effect associated with closing an item.
   *
   * Used when closing tabs, windows, or connections.
   */
  static void Close();

  /**
   * @brief Plays the primary error sound effect.
   *
   * Used to indicate a general failure or invalid operation.
   */
  static void Error_1();

  /**
   * @brief Plays the secondary error sound effect.
   *
   * Used for alternative error conditions or critical warnings.
   */
  static void Error_2();

  /**
   * @brief Plays the application exit sound effect.
   */
  static void Exit();

  /**
   * @brief Plays the sound effect associated with opening an item.
   *
   * Used when opening files, connections, or new windows.
   */
  static void Open();

  /**
   * @brief Plays the refresh sound effect.
   *
   * Triggered when reloading data or refreshing a view.
   */
  static void Refresh();

  /**
   * @brief Plays the primary execution sound effect.
   *
   * Typically used when starting a query execution or a long-running process.
   */
  static void Run_1();

  /**
   * @brief Plays the secondary execution sound effect.
   *
   * An alternative sound for process execution.
   */
  static void Run_2();

  /**
   * @brief Plays a camera shutter sound effect.
   *
   * Used when capturing a screenshot or saving a visual report.
   */
  static void Screenshot();

};

#endif // DSOUNDEFFECT_H
