/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DSETTINGS_H
#define DSETTINGS_H

#include <QSettings>

/**
 * @brief The DSettings class manages application configuration and persistence.
 *
 * This class extends QSettings to provide a standardized interface for storing
 * and retrieving application settings. It abstracts specific organization/application
 * names to ensure consistency across the application.
 */
class DSettings : public QSettings
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the DSettings object.
   *
   * Initializes the settings mechanism using the default application and organization
   * configuration defined in the main application instance.
   */
  DSettings();

  /**
   * @brief Sets the value of a setting.
   * @param key The name of the setting.
   * @param value The value to be stored.
   */
  void setValue(QAnyStringView key, const QVariant &value);

  /**
   * @brief Retrieves the value of a setting with a fallback default.
   * @param key The name of the setting.
   * @param defaultValue The value to return if the setting does not exist.
   * @return The stored value converted to QVariant, or defaultValue if not found.
   */
  QVariant value(QAnyStringView key, const QVariant &defaultValue) const;

  /**
   * @brief Retrieves the value of a setting.
   * @param key The name of the setting.
   * @return The stored value converted to QVariant.
   */
  QVariant value(QAnyStringView key) const;

  /**
   * @brief Retrieves the path to the configuration file.
   * @return A QString containing the absolute path to the settings file.
   */
  QString fileName() const;

//  void beginGroup(QAnyStringView prefix);
//  void endGroup();
//  QString group() const;

  /**
   * @brief Retrieves all keys within a specific group.
   * @param groupName The name of the group to inspect.
   * @return A list of keys contained within the specified group.
   */
  QStringList allKeys(QAnyStringView groupName);

  /**
   * @brief Removes all settings within a specific group.
   * @param groupName The name of the group to clear.
   */
  void clearGroup(QAnyStringView groupName);

  /**
   * @brief Removes a specific setting.
   * @param key The name of the setting to remove.
   */
  void remove(QAnyStringView key);

  /**
   * @brief Writes any unsaved changes to permanent storage.
   *
   * Ensures that the internal cache is flushed to the disk or registry.
   */
  void sync();
};

#endif // DSETTINGS_H
