/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DREPORTVIEWER_H
#define DREPORTVIEWER_H

#include <QSystemTrayIcon>
#include <QtCharts/QChartView>
#include <QtCharts/QPieSeries>

#include "dmdisubwindow.h"
#include "dbms.h"
#include "dsettings.h"

/**
 * @brief Helper class defining available report visualization types.
 */
class ReportTypes
{
public:
  /**
   * @brief Enumeration representing the visual format of the report.
   */
  enum ReportType {PlainText, BarChart, PieChart, None};
};

class QPlainTextEdit;
class QPushButton;
class DPieChartWidget;
class QComboBox;
class QLineEdit;
class BaseTextEditor;
class QCompleter;

///QT_CHARTS_USE_NAMESPACE - Line commented due to Qt6

/**
 * @brief The DReportViewer class displays database reports in various formats.
 *
 * This class inherits from DMdiSubWindow and is responsible for executing SQL
 * queries and presenting the results as either plain text, bar charts, or pie charts.
 * It also supports exporting these reports to PDF, Image, or TXT formats.
 */
class DReportViewer : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the DReportViewer window.
   * @param serverConnection Pointer to the active DBMS connection.
   * @param reportTitle The title displayed on the window and report.
   * @param reportType The visualization type (e.g., PieChart, PlainText).
   * @param unit The unit of measurement for chart values (optional).
   */
  DReportViewer(DBMS *serverConnection, QString reportTitle, ReportTypes::ReportType reportType, QString unit = QString());

  /**
   * @brief Sets the SQL query used to generate the report.
   * @param sqlQuery The SQL select statement.
   */
  void setSqlQuery(QString sqlQuery);

  /**
   * @brief Retrieves the current SQL query.
   * @return A QString containing the SQL statement.
   */
  QString getSqlQuery();

  /**
   * @brief Initiates the process to add a user-defined custom report.
   */
  void addCustomReport();

public slots:
  /**
   * @brief Executes the query and populates the report view.
   *
   * Fetches data from the DBMS and renders it according to the selected ReportType.
   */
  void showReportData();

private:
  DBMS *serverConnection;
  ReportTypes::ReportType reportType;
  QPlainTextEdit *plainTextReport;
  QPushButton *pushButtonExportToPDF;
  DSettings settings;
  QString sqlQuery;
  QPushButton *pushButtonRefresh;
  DPieChartWidget *dPieChart;
  QList<QStringList> *result;
  QPushButton *pushButtonExportToImage;
  QComboBox *comboReportName;
  QLineEdit *lineEditUnit;
  QComboBox *comboReportType;
  BaseTextEditor *baseTextEditor;
  QCompleter *completer;
  QDialog *dialog;
  QPushButton *pushButtonExportToTXT;
  QPieSeries *series;
  QChart *chart;
  QChartView *chartView;
  QString unit;
  QString reportTitle;

  /**
   * @brief Retranslates the user interface.
   *
   * Updates titles, tooltips, and button text for language changes.
   */
  void retranslateUI();

private slots:
  /**
   * @brief Exports the current report content to a PDF file.
   */
  void pushButtonExportToPDFClicked();

  /**
   * @brief Exports the current chart to an image file (PNG/JPG).
   */
  void pushButtonExportToImageClicked();

  /**
   * @brief Configures the custom report widgets based on a selected profile.
   * @param reportName The name of the saved report configuration.
   */
  void fillCustomReportwidgets(QString reportName);

  /**
   * @brief Discards the current custom report configuration or changes.
   */
  void discardReportSlot();

  /**
   * @brief Exports the current text report to a plain text file.
   */
  void pushButtonExportToTXTClicked();

  /**
   * @brief Displays help information for the report viewer.
   */
  void helpSlot();

signals:
  /**
   * @brief Emitted to display a status message in the main application.
   * @param message The text message.
   * @param icon The icon type (Information, Warning, etc.).
   * @param timeout Duration to display the message (ms).
   */
  void statusBarMessage(const QString &message, QSystemTrayIcon::MessageIcon icon = QSystemTrayIcon::Information, int timeout = 0);
};

#endif // DREPORTVIEWER_H
