/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QVBoxLayout>
#include <QPlainTextEdit>
#include <QPushButton>
#include <QFileDialog>
#include <QApplication>
#include <QPrinter>
#include <QImageWriter>
#include <QDialogButtonBox>
#include <QComboBox>
#include <QFormLayout>
#include <QLineEdit>
#include <QCompleter>

#include "dreportviewer.h"
#include "dtitlelabel.h"
#include "staticfunctions.h"
//#include "dpiechartwidget.h"
#include "basetexteditor.h"
#include "dicon.h"

#include <QDebug>

DReportViewer::DReportViewer(DBMS *serverConnection, QString reportTitle, ReportTypes::ReportType reportType, QString unit)
{
//  Q_UNUSED(unit); //Delete this line if the parameter is used againg
  this->serverConnection = serverConnection;
  this->reportType = reportType;
  this->unit = unit;
  this->reportTitle = reportTitle;
  setWindowTitle(tr("Report:") + " " + reportTitle);
  setObjectName(windowTitle());

  setWindowIcon((reportType == ReportTypes::PlainText ? DIcon::TextEditor() : DIcon::GraphicsCharts()));
  QVBoxLayout *mainVLayout = new QVBoxLayout;
  mainVLayout->setContentsMargins(3, 0, 3, 0);
  mainVLayout->addWidget(new DTitleLabel(windowTitle()));

  QHBoxLayout *groupBoxHLayout = new QHBoxLayout;

  pushButtonRefresh = new QPushButton;
  pushButtonRefresh->setIcon(DIcon::Refresh());
  connect(pushButtonRefresh, SIGNAL(clicked()), this, SLOT(showReportData()));
  groupBoxHLayout->addWidget(pushButtonRefresh);

  pushButtonExportToPDF = new QPushButton;
  pushButtonExportToPDF->hide();
  pushButtonExportToPDF->setIcon(DIcon::PDF());
  connect(pushButtonExportToPDF, SIGNAL(clicked()), this, SLOT(pushButtonExportToPDFClicked()));
  groupBoxHLayout->addWidget(pushButtonExportToPDF);

  pushButtonExportToTXT = new QPushButton;
  pushButtonExportToTXT->hide();
  pushButtonExportToTXT->setIcon(DIcon::TextEditor());
  connect(pushButtonExportToTXT, SIGNAL(clicked()), this, SLOT(pushButtonExportToTXTClicked()));
  groupBoxHLayout->addWidget(pushButtonExportToTXT);

  pushButtonExportToImage = new QPushButton;
  pushButtonExportToImage->hide();
  pushButtonExportToImage->setIcon(DIcon::ImagePNG());
  connect(pushButtonExportToImage, SIGNAL(clicked()), this, SLOT(pushButtonExportToImageClicked()));
  groupBoxHLayout->addWidget(pushButtonExportToImage);

  groupBoxHLayout->addStretch(1);
  mainVLayout->addLayout(groupBoxHLayout);

  switch (reportType) {
  case ReportTypes::PlainText:
    plainTextReport = new QPlainTextEdit;
    plainTextReport->setFont(StaticFunctions::fixedWidthFont());
    plainTextReport->setWordWrapMode(QTextOption::NoWrap);
    mainVLayout->addWidget(plainTextReport);
    pushButtonExportToPDF->show();
    pushButtonExportToTXT->show();
    break;
  case ReportTypes::PieChart:
    series = new QPieSeries();
    chart = new QChart();
    chart->legend()->hide();
    chartView = new QChartView();
    chart->addSeries(series);
    chartView->setChart(chart);
    chartView->setRenderHint(QPainter::Antialiasing);
    // See revision 246 for full implementation of DPieWidget
//    dPieChart = new DPieChartWidget(unit);
    mainVLayout->addWidget(chartView);
    pushButtonExportToImage->show();
    break;
  case ReportTypes::BarChart:
  case ReportTypes::None:
  default:
    break;
  }

  QWidget *widMain = new QWidget;
  retranslateUI();
  widMain->setLayout(mainVLayout);
  setWidget(widMain);
}

void DReportViewer::setSqlQuery(QString sqlQuery)
{
  this->sqlQuery = sqlQuery;
}

QString DReportViewer::getSqlQuery()
{
  return sqlQuery;
}

void DReportViewer::addCustomReport()
{
  dialog = new QDialog(this);

  QVBoxLayout *verticalLayout = new QVBoxLayout;
  verticalLayout->addWidget(new DTitleLabel(windowTitle()));

  comboReportName = new QComboBox();
  connect(comboReportName, SIGNAL(textActivated(QString)), this, SLOT(fillCustomReportwidgets(QString)));
  comboReportName->setEditable(true);
  QStringList reports = settings.allKeys("CustomReports");
  foreach (QString report, reports)
    comboReportName->addItem(report);

  completer = new QCompleter(reports, comboReportName);
  completer->setFilterMode(Qt::MatchContains);
  completer->setCaseSensitivity(Qt::CaseInsensitive);
  comboReportName->setCompleter(completer);

  lineEditUnit = new QLineEdit;

  comboReportType = new QComboBox();
  //PlainText, BarChart, PieChart
  comboReportType->addItem(tr("Plain text"), ReportTypes::PlainText);
  //comboReportType->addItem(tr("Bar chart"), ReportTypes::BarChart);
  comboReportType->addItem(tr("Pie chart"), ReportTypes::PieChart);

  baseTextEditor = new BaseTextEditor(EditorTypes::SQLQuery);

  QFormLayout *formLayout = new QFormLayout;
  formLayout->addRow(tr("&Report Name:"), comboReportName);
  formLayout->addRow(tr("&Unit:"), lineEditUnit);
  formLayout->addRow(tr("&Type:"), comboReportType);
  formLayout->addRow(baseTextEditor);
  verticalLayout->addLayout(formLayout);

  QDialogButtonBox *buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel | QDialogButtonBox::Discard | QDialogButtonBox::Help);
  connect(buttonBox, SIGNAL(accepted()), dialog, SLOT(accept()));
  connect(buttonBox, SIGNAL(rejected()), dialog, SLOT(reject()));
  connect(buttonBox->button(QDialogButtonBox::Discard), SIGNAL(clicked()), this, SLOT(discardReportSlot()));
  connect(buttonBox->button(QDialogButtonBox::Help), SIGNAL(clicked()), this, SLOT(helpSlot()));
  verticalLayout->addWidget(buttonBox);

  fillCustomReportwidgets(comboReportName->currentText());
  comboReportName->lineEdit()->selectAll();

  dialog->setLayout(verticalLayout);
  if (dialog->exec() == QDialog::Accepted) {
    settings.setValue("CustomReports/" + comboReportName->currentText(), QString("#Unit:%1#Type:%2#SQLQuery:%3")
                      .arg(lineEditUnit->text()
                      , comboReportType->currentData().toString()
                      , baseTextEditor->toPlainText()));
  }
}

void DReportViewer::showReportData()
{
  switch (reportType) {
  case ReportTypes::PlainText:
    if (plainTextReport) {
      plainTextReport->setPlainText(serverConnection->outputAsTable(sqlQuery));
    }
    break;
  case ReportTypes::PieChart: {
    QApplication::setOverrideCursor(Qt::WaitCursor);
    series->clear();
//      dPieChart->clear();
    result = serverConnection->runQuery(sqlQuery);
    result->takeLast(); //Remove the "Affected rows" line.
    for (int counter = 0; counter < result->count(); counter++) {
      series->append(result->at(counter).at(0) + ": " + result->at(counter).at(1), result->at(counter).at(1).toDouble());
      series->slices().at(counter)->setLabelVisible(true);

//        series->slices().at(counter)->setLabel(result->at(counter).at(1));
    }
    QApplication::restoreOverrideCursor();
  }
    break;
  default:
    break;
  }
}

void DReportViewer::retranslateUI()
{
  pushButtonExportToPDF->setText(tr("Export to PDF"));
  pushButtonExportToPDF->setToolTip(pushButtonExportToPDF->text());
  pushButtonRefresh->setText(tr("Refresh"));
  pushButtonRefresh->setToolTip(pushButtonRefresh->text());
  pushButtonExportToImage->setText(tr("Export to Image"));
  pushButtonExportToImage->setToolTip(pushButtonExportToImage->text());
  pushButtonExportToTXT->setText(tr("Export to TXT"));
  pushButtonExportToTXT->setToolTip(pushButtonExportToTXT->text());
  if (reportType == ReportTypes::PieChart)
    chart->setTitle(reportTitle + ".\n" + tr("Unit: ") + unit);
}

void DReportViewer::pushButtonExportToPDFClicked()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QFileDialog fileDialog;
  fileDialog.setDirectory(QDir::home());
  QString file = fileDialog.getSaveFileName(this, tr("Save to Pdf"), settings.value("CustomReports/LastFilePdf", QDir::home().absolutePath()).toString(), tr("Pdf & Ps files (*.pdf *.ps)"));
  settings.setValue("CustomReports/LastFilePdf", fileDialog.directory().filePath(file));
  QPrinter printer(QPrinter::HighResolution);
  printer.setOutputFileName(file);
  printer.setOutputFormat(file.endsWith(".pdf") ? QPrinter::PdfFormat : QPrinter::NativeFormat);
  plainTextReport->document()->print(&printer);
  QApplication::restoreOverrideCursor();
  emit statusBarMessage(tr("File saved at: %1").arg(file));
}

void DReportViewer::pushButtonExportToImageClicked()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QFileDialog fileDialog;
  fileDialog.setDirectory(QDir::home());
  QString imageFormats;
  foreach(QString format, QImageWriter::supportedImageFormats())
    imageFormats += "*." + format + " ";
  QString file = fileDialog.getSaveFileName(this, tr("Save to Image"), settings.value("CustomReports/LastFileImg", QDir::home().absolutePath()).toString(), tr("Image files (%1)").arg(imageFormats));
  settings.setValue("CustomReports/LastFileImg", fileDialog.directory().filePath(file));
  QPixmap pixmap(chartView->grab());
//  dPieChart->render(&pixmap);
  pixmap.save(file, "PNG", 5);
  QApplication::restoreOverrideCursor();
  emit statusBarMessage(tr("File saved at: %1").arg(file));
}

void DReportViewer::fillCustomReportwidgets(QString reportName)
{
  QStringList data = settings.value("CustomReports/" + reportName).toString().split("#", Qt::SkipEmptyParts);
  if (data.count() > 0) {
    lineEditUnit->setText(data.at(0).split(":").at(1));
    comboReportType->setCurrentIndex(comboReportType->findData(data.at(1).split(":").at(1).toInt()));
    baseTextEditor->setPlainText(data.at(2).split(":").at(1));
  }
}

void DReportViewer::discardReportSlot()
{
  settings.remove("CustomReports/" + comboReportName->currentText());
  dialog->close();
}

void DReportViewer::pushButtonExportToTXTClicked()
{
  QString fileName = QFileDialog::getSaveFileName(this, tr("Save file"), QDir::home().absolutePath(), tr("Text files (%1)").arg("txt"));
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QFile file(fileName);
  if (!file.open(QFile::WriteOnly | QFile::Text))
    emit statusBarMessage(tr("Cannot write file %1:\n%2.").arg(fileName, file.errorString()), QSystemTrayIcon::Critical);
  QTextStream out(&file);
  out << plainTextReport->toPlainText();
  file.close();
  emit statusBarMessage(tr("File saved at: %1").arg(fileName), QSystemTrayIcon::Information);
  QApplication::restoreOverrideCursor();
}

void DReportViewer::helpSlot()
{
  comboReportName->addItem("Example report");
  lineEditUnit->setText("Units");
  comboReportType->setCurrentIndex(comboReportType->findText(tr("Pie chart")));
  baseTextEditor->setPlainText("SELECT 'Value1', 10"
                               "\nUNION"
                               "\nSELECT 'Value2', 20"
                               "\nUNION"
                               "\nSELECT 'Value3', 30;");
}
