/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DQUERYLOGGER_H
#define DQUERYLOGGER_H

#include <QSettings>
#include <QtSql/QSqlDatabase>
#include <QObject>

#include "dbms.h"
#include "dsettings.h"

/**
 * @brief The DQueryLogger class handles the recording of executed SQL statements.
 *
 * This class is responsible for maintaining a history of queries run by the user.
 * It primarily uses a local SQLite database to store the logs, allowing for persistent
 * history retrieval even after the application closes.
 */
class DQueryLogger : public QObject
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the DQueryLogger.
   * @param serverConnection Pointer to the main DBMS connection associated with this logger.
   */
  DQueryLogger(DBMS *serverConnection);

  /**
   * @brief Clears the local query history.
   *
   * Deletes all records from the local SQLite log database.
   */
  void clearSQLiteQueryLog();

  /**
   * @brief Logs a specific query and its result.
   *
   * This is the high-level method called when a query finishes execution.
   *
   * @param query The SQL statement that was executed.
   * @param result The outcome of the execution (e.g., "OK", error message, or row count).
   */
  void logExecutedQueries(QString query, QString result);

  /**
   * @brief Inserts a detailed record into the log database.
   *
   * @param sessionid The unique identifier for the current database session.
   * @param connection The name of the connection profile used.
   * @param query The SQL statement.
   * @param result The outcome of the execution.
   */
  void insertOnExecutedQueries(QString sessionid, QString connection, QString query, QString result);

  /**
   * @brief Retrieves the raw MariaDB connection handle used for logging.
   * @return A pointer to the MYSQL structure.
   */
  MYSQL *getMariadbConnection();

//signals:
//  void error(QString errorText, QString errorType);

private:
  DBMS *serverConnection;
  DBMS *remoteLogginServerConnection;
  QSqlDatabase dbSQLite;
  DSettings settings;

};

#endif // DQUERYLOGGER_H
