/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QtSql/QSqlQuery>
#include <QtSql/QSqlError>
#include <QErrorMessage>
#include <QRegularExpression>
#include <QDir>
#include <QMessageBox>
#include <QMap>
#include <QApplication>

#include "dquerylogger.h"
#include "staticfunctions.h"

#include <QDebug>

DQueryLogger::DQueryLogger(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
  QFileInfo settingsInfo(settings.fileName());
  qApp->setProperty("LocalDatabase", settingsInfo.absoluteDir().absolutePath() + QDir::separator() + QCoreApplication::applicationName() + ".sqlite");
  dbSQLite = QSqlDatabase::addDatabase("QSQLITE");
  dbSQLite.setDatabaseName(qApp->property("LocalDatabase").toString());
  if (!dbSQLite.open()) {
    QMessageBox::critical(0, tr("Cannot open database"), tr("Unable to establish a database connection for the Query Log.")
                          + "\n" + tr("Error: ") + dbSQLite.lastError().text()
                          + "\n" + tr("File: ") + qApp->property("LocalDatabase").toString()
                          , QMessageBox::Ok);
  }
  if (dbSQLite.open()) {
    QSqlQuery querySQLite;
    if (!querySQLite.exec("CREATE TABLE IF NOT EXISTS executedqueries (LineNumber INTEGER, SessionId TEXT, Date TEXT, Connection TEXT, Query TEXT, Result TEXT, PRIMARY KEY(LineNumber DESC))"))
      this->serverConnection->errorMessage->showMessage(querySQLite.lastError().text(), "LogExecutedQueries");
    querySQLite.exec("CREATE INDEX `index1` ON `executedqueries` ( `Query` ASC )");
    querySQLite.exec("CREATE INDEX `index2` ON `executedqueries` ( `Date` DESC )");
  }
  remoteLogginServerConnection = new DBMS(false);
  if (settings.value("GeneralSettings/QueryLogOnRemoteDatabase", "None").toString() != "None") {
    QMap<QString, QVariant> params = settings.value("ServerConnections/" + settings.value("GeneralSettings/QueryLogOnRemoteDatabase", "None").toString()).toMap();
//    remoteLogginServerConnection->close();
    remoteLogginServerConnection->setUserName(params.value("User").toString());
    remoteLogginServerConnection->setHostName(params.value("Host").toString());
    remoteLogginServerConnection->setPassword(StaticFunctions::password(params.value("Password").toString()));
    //remoteLogginServerConnection->setDatabase("mysql"); //Connect to mysql in case that Caliope does no exists
    remoteLogginServerConnection->setPort(params.value("Port").toInt());
    remoteLogginServerConnection->setCharacterSet(params.value("Collation").toString().split("|").at(0));
    remoteLogginServerConnection->setCollation(params.value("Collation").toString().split("|").at(1));
    remoteLogginServerConnection->setUseSSL(params.value("UseSSL").toInt());
    remoteLogginServerConnection->setKeyFile(params.value("KeyFile").toString());
    remoteLogginServerConnection->setCertFile(params.value("CertFile").toString());
    if (params.value("ConnectionType") == "--")
      remoteLogginServerConnection->setDBMSType(StaticFunctions::Undefined);
    if (params.value("ConnectionType")  == "MySQL")
      remoteLogginServerConnection->setDBMSType(StaticFunctions::MySQL);
    if (params.value("ConnectionType")  == "MariaDB")
      remoteLogginServerConnection->setDBMSType(StaticFunctions::MariaDB);
    if (!remoteLogginServerConnection->open()) {
      QMessageBox::critical(0, tr("Cannot connect to the remote loggin server"), remoteLogginServerConnection->lastError());
    }
    remoteLogginServerConnection->runQuery("CREATE DATABASE IF NOT EXISTS Caliope");
    remoteLogginServerConnection->runQuery("USE Caliope");
    //remoteLogginServerConnection->runQuery("DROP TABLE Caliope.executedqueries");
    remoteLogginServerConnection->runQuery("CREATE TABLE IF NOT EXISTS Caliope.executedqueries (LineNumber INTEGER AUTO_INCREMENT, SessionId TEXT, Date TEXT, Connection TEXT, Query TEXT, Result TEXT, PRIMARY KEY(LineNumber DESC), KEY `index1` (`Query` ASC), KEY `index2` (`Date` DESC)) COLLATE utf8mb4_spanish_ci");
  }
}

void DQueryLogger::clearSQLiteQueryLog()
{
  if (settings.value("GeneralSettings/EnableQueryLog", false).toBool()) {
    if (dbSQLite.open()) {
      QSqlQuery querySQLite;
      if (!querySQLite.exec("DELETE FROM executedqueries"))
        this->serverConnection->errorMessage->showMessage(querySQLite.lastError().text(), "LogExecutedQueries");
    }
    if (remoteLogginServerConnection->isOpened()) {
      remoteLogginServerConnection->runQuery("DELETE FROM Caliope.executedqueries");
    }
  }
}

void DQueryLogger::logExecutedQueries(QString query, QString result)
{
  if (settings.value("GeneralSettings/EnableQueryLog", false).toBool()) {
    if (!dbSQLite.isOpen())
      dbSQLite.open();
    if (dbSQLite.isOpen()) {
      QSqlQuery querySQLite;
      querySQLite.prepare("INSERT INTO executedqueries (sessionid, date, connection, query, result) VALUES (:sessionid, :date, :connection, :query, :result)");
      querySQLite.bindValue(":sessionid", qApp->property("SessionId").toString());
      querySQLite.bindValue(":date", QDateTime::currentDateTime());
      querySQLite.bindValue(":connection", this->serverConnection->getConnectionString());
      querySQLite.bindValue(":query", query);
      querySQLite.bindValue(":result", result.mid(0, 5000));
      if (!querySQLite.exec())
        this->serverConnection->errorMessage->showMessage(querySQLite.lastError().text(), "LogExecutedQueries");
    }
    if (remoteLogginServerConnection->isOpened()) {
      insertOnExecutedQueries(qApp->property("SessionId").toString(), this->serverConnection->getConnectionString(), query, result);
    }
  }
}

void DQueryLogger::insertOnExecutedQueries(QString sessionid, QString connection, QString query, QString result)
{
  if (settings.value("GeneralSettings/EnableQueryLog", false).toBool()) {
    if (!dbSQLite.isOpen())
      dbSQLite.open();
    if (dbSQLite.isOpen()) {
      QSqlQuery querySQLite;
      querySQLite.prepare("INSERT INTO executedqueries (sessionid, date, connection, query, result) VALUES (:sessionid, :date, :connection, :query, :result)");
      querySQLite.bindValue(":sessionid", sessionid);
      querySQLite.bindValue(":date", QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss.zzz"));
      querySQLite.bindValue(":connection", connection);
      querySQLite.bindValue(":query", query);
      querySQLite.bindValue(":result", result.mid(0, 5000));
      if (!querySQLite.exec())
        this->serverConnection->errorMessage->showMessage(querySQLite.lastError().text(), "LogExecutedQueries");
    }
    if (remoteLogginServerConnection->isOpened()) {
      // https://mariadb.com/kb/en/bulk-insert-column-wise-binding/
//      MYSQL_STMT *stmt;
//      MYSQL_BIND bind[3];

//      /* Data for insert */
//      const char *surnames[]= {"Milky Way", "Axmark", "N.N.", "David"};
//      unsigned long surnames_length[]= {8,6,4,5};
//      const char *forenames[]= {"Monty", "David", "will be replaced by default value", "Villalobos"};
//      char forename_ind[]= {STMT_INDICATOR_NTS, STMT_INDICATOR_NTS, STMT_INDICATOR_DEFAULT, STMT_INDICATOR_NTS};
//      char id_ind[]= {STMT_INDICATOR_NULL, STMT_INDICATOR_NULL, STMT_INDICATOR_NULL, STMT_INDICATOR_NULL};
//      unsigned int array_size= 4;

//      stmt= mysql_stmt_init(remoteLogginServerConnection->getMariadbConnection());
//      if (mysql_stmt_prepare(stmt, "INSERT INTO Caliope.bulk_example1 VALUES (?,?,?)", -1))
//        qDebug() << mysql_stmt_errno(stmt) << mysql_stmt_sqlstate(stmt) << mysql_stmt_error(stmt);

//      memset(bind, 0, sizeof(MYSQL_BIND) * 3);

//      /* We autogenerate id's, so all indicators are STMT_INDICATOR_NULL */
//      bind[0].u.indicator= id_ind;
//      bind[0].buffer_type= MYSQL_TYPE_LONG;

//      bind[1].buffer= forenames;
//      bind[1].buffer_type= MYSQL_TYPE_STRING;
//      bind[1].u.indicator= forename_ind;

//      bind[2].buffer_type= MYSQL_TYPE_STRING;
//      bind[2].buffer= surnames;
//      bind[2].length= surnames_length;

//      /* set array size */
//      mysql_stmt_attr_set(stmt, STMT_ATTR_ARRAY_SIZE, &array_size);

//      /* bind parameter */
//      mysql_stmt_bind_param(stmt, bind);

//      /* execute */
//      if (mysql_stmt_execute(stmt))
//        qDebug() << mysql_stmt_errno(stmt) << mysql_stmt_sqlstate(stmt) << mysql_stmt_error(stmt);

//      mysql_stmt_close(stmt);
//      //  mysql_close(mysql);
      /**********************/
      MYSQL_STMT *stmt;
      MYSQL_BIND bind[5]; //(sessionid, date, connection, query, result)

      char uIndicator[] = {STMT_INDICATOR_NTS};
      unsigned int totalRowsToInsert = 1;

      stmt= mysql_stmt_init(remoteLogginServerConnection->getMariadbConnection());
      if (mysql_stmt_prepare(stmt, "INSERT INTO Caliope.executedqueries (sessionid, date, connection, query, result) VALUES (?, ?, ?, ?, ?)", -1))
        qDebug() << mysql_stmt_errno(stmt) << mysql_stmt_sqlstate(stmt) << mysql_stmt_error(stmt);

      memset(bind, 0, sizeof(MYSQL_BIND) * totalRowsToInsert);

      //const char *sessionData[] = {sessionid.toUtf8().data()};
      const char *sessionData[] = {QString("%1-1").arg(sessionid).toUtf8().data()}; //Added the '-1' because otherwise 'QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss.zzz").toUtf8().data()' is inserted, and yes, I do not know why
      bind[0].buffer = sessionData;
      bind[0].buffer_type = MYSQL_TYPE_STRING;
      bind[0].u.indicator = uIndicator;

      const char *dateData[] = {QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss.zzz").toUtf8().data()};
      bind[1].buffer = dateData;
      bind[1].buffer_type = MYSQL_TYPE_STRING;
      bind[1].u.indicator = uIndicator;

      const char *connectionData[] = {connection.toUtf8().data()};
      bind[2].buffer = connectionData;
      bind[2].buffer_type = MYSQL_TYPE_STRING;
      bind[2].u.indicator = uIndicator;

      const char *executedQuery[] = {query.toUtf8().data()};
      bind[3].buffer = executedQuery;
      bind[3].buffer_type = MYSQL_TYPE_STRING;
      bind[3].u.indicator = uIndicator;

      const char *queryResult[] = {result.mid(0, 5000).toUtf8().data()};
      bind[4].buffer = queryResult;
      bind[4].buffer_type = MYSQL_TYPE_STRING;
      bind[4].u.indicator = uIndicator;

      mysql_stmt_attr_set(stmt, STMT_ATTR_ARRAY_SIZE, &totalRowsToInsert);

      mysql_stmt_bind_param(stmt, bind);

      if (mysql_stmt_execute(stmt))
        qDebug() << mysql_stmt_errno(stmt) << mysql_stmt_sqlstate(stmt) << mysql_stmt_error(stmt);

      mysql_stmt_close(stmt);
    }
  }
}

MYSQL *DQueryLogger::getMariadbConnection()
{
  return this->remoteLogginServerConnection->getMariadbConnection();
}
