/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DMONITORING_H
#define DMONITORING_H

#include <QDialog>
#include <QSystemTrayIcon>

#include "dmdisubwindow.h"
#include "dsettings.h"

class DTitleLabel;
class QPushButton;
class QTextBrowser;
class QSpinBox;
class QGroupBox;
class DTableView;
class QAbstractButton;
class DBMS;
class QLabel;

/**
 * @brief Dialog window for configuring which connections are monitored.
 *
 * This class provides a table interface to add, remove, or modify the list
 * of database connections that the monitoring system will check.
 */
class MonitoredConnection : public QDialog
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the MonitoredConnection dialog.
   */
  MonitoredConnection();

private:
  DTableView *dTableView;
  QList<QStringList> *headers;
  DSettings settings;

  /**
   * @brief Loads the list of saved connections from settings into the table view.
   */
  void loadMonitoredConnections();

private slots:
  /**
   * @brief Saves the current list of connections from the table view to persistent storage.
   */
  void saveMonitoredConnections();

  /**
   * @brief Handles standard dialog button clicks (e.g., Save, Cancel).
   * @param button The button that was clicked.
   */
  void actionRoleSlot(QAbstractButton *button);

  /**
   * @brief Displays help information for this dialog.
   */
  void helpSlot();

signals:
  /**
   * @brief Emitted when the configuration changes, requesting the main monitor to refresh.
   */
  void reloadData();

  /**
   * @brief Emitted to request the help viewer.
   */
  void showHelp();
};

/**
 * @brief The DMonitoring class serves as the main dashboard for system monitoring.
 *
 * This window performs periodic checks on the configured database connections,
 * evaluating specific fields against defined thresholds, and displaying logs
 * or alerts in text browsers.
 */
class DMonitoring : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the DMonitoring window.
   */
  DMonitoring();

  /**
   * @brief Retranslates the user interface.
   *
   * Updates labels and button text to reflect the current language.
   */
  void retranslateUI();

private:
  DTitleLabel *dTitleLabel;
  QPushButton *pushButtonSettings;
  QTextBrowser *qTextBrowser1;
  QTextBrowser *qTextBrowser2;
  QTimer *timerRefresh;
  QTimer *secondsToNextIterationTimer;
  DSettings settings;
  QSpinBox *spinBoxRefreshRate;
  QPushButton *pushButtonStopRefreshing;
  QGroupBox *buttonGroup;
  unsigned int interationsCount;
  MonitoredConnection *monitoredConnection;
  QPushButton *pushButtonReloadData;
  DBMS *serverConnection;
  QLabel *secondsToNextIteration;
  QPushButton *pushButtonShowHelp;

  /**
   * @brief Evaluates a monitoring condition.
   *
   * Compares a value retrieved from the database against a threshold using a specified operator.
   *
   * @param resultField The actual value obtained from the query.
   * @param comparition The comparison operator (e.g., ">", "<", "=").
   * @param defaultValue The threshold value to compare against.
   * @return True if the condition is met (triggering an alert/log), false otherwise.
   */
  bool compareResultField(QString resultField, QString comparition, QString defaultValue);

public slots:
  /**
   * @brief Displays help information regarding the monitoring tool.
   */
  void helpSlot();

private slots:
  /**
   * @brief Executes the monitoring iteration.
   *
   * Connects to servers, runs checks, and updates the text browsers with results.
   */
  void reloadData();

  /**
   * @brief Updates the refresh interval based on user input.
   * @param value The new interval in seconds.
   */
  void refreshRateSlot(const int value);

  /**
   * @brief Opens the MonitoredConnection settings dialog.
   */
  void pushButtonSettingsSlot();

  /**
   * @brief Updates the countdown label showing seconds until the next check.
   */
  void secondsToNextIterationSlot();

signals:
  /**
   * @brief Emitted to update the progress bar during a check iteration.
   * @param progress The percentage complete (0-100).
   */
  void loadProgress(const int progress);

  /**
   * @brief Emitted to display a status message in the system tray or status bar.
   * @param message The message text.
   * @param icon The icon type (Information, Warning, Error).
   * @param timeout Duration to show the message.
   */
  void statusBarMessage(const QString &message, QSystemTrayIcon::MessageIcon icon = QSystemTrayIcon::Information, int timeout = 0);
};

#endif // DMONITORING_H
