/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QHBoxLayout>
#include <QPushButton>
#include <QTextBrowser>
#include <QTimer>
#include <QSpinBox>
#include <QGroupBox>
#include <QDialogButtonBox>
#include <QStandardItem>
#include <QApplication>
#include <QMessageBox>
#include <QScrollBar>

#include "dmonitoring.h"
#include "dicon.h"
#include "dtitlelabel.h"
#include "dtableview.h"
#include "staticfunctions.h"
#include "dbms.h"

#include <QDebug>

DMonitoring::DMonitoring()
{
  setWindowIcon(DIcon::DatabaseMonitorig());
  timerRefresh = new QTimer(this);
  connect(timerRefresh, SIGNAL(timeout()), this, SLOT(reloadData()));
  secondsToNextIterationTimer = new QTimer(this);
  connect(secondsToNextIterationTimer, SIGNAL(timeout()), this, SLOT(secondsToNextIterationSlot()));
  serverConnection = new DBMS(false);
  timerRefresh->setInterval(1000 * settings.value("Monitoring/RefreshRate", 1).toInt());
  timerRefresh->start();
  interationsCount = 0;
  QWidget *widMain = new QWidget;
  QVBoxLayout *mainVLayout = new QVBoxLayout;
  mainVLayout->setContentsMargins(3, 0, 3, 0);
  dTitleLabel = new DTitleLabel;
  buttonGroup = new QGroupBox(this);
  mainVLayout->addWidget(dTitleLabel);
  QHBoxLayout *qHBoxLayout = new QHBoxLayout;
  pushButtonSettings = new QPushButton;
  pushButtonSettings->setIcon(DIcon::DatabaseServer());
  connect(pushButtonSettings, SIGNAL(clicked()), this, SLOT(pushButtonSettingsSlot()));
  qHBoxLayout->addWidget(pushButtonSettings);
  spinBoxRefreshRate = new QSpinBox;
  spinBoxRefreshRate->setRange(0, 2147483647);
  qHBoxLayout->addWidget(spinBoxRefreshRate);
  spinBoxRefreshRate->setValue(settings.value("Monitoring/RefreshRate", 1).toUInt());
  connect(spinBoxRefreshRate, SIGNAL(valueChanged(int)), this, SLOT(refreshRateSlot(int)));
  secondsToNextIteration = new QLabel;
  qHBoxLayout->addWidget(secondsToNextIteration);
  pushButtonStopRefreshing = new QPushButton;
  pushButtonStopRefreshing->setIcon(DIcon::Stop());
  pushButtonStopRefreshing->setCheckable(true);
  pushButtonStopRefreshing->setChecked(false);
  connect(pushButtonStopRefreshing, SIGNAL(toggled(bool)), timerRefresh, SLOT(start()));
  qHBoxLayout->addWidget(pushButtonStopRefreshing);
  pushButtonReloadData = new QPushButton;
  pushButtonReloadData->setIcon(DIcon::Refresh());
  connect(pushButtonReloadData, SIGNAL(clicked()), this, SLOT(reloadData()));
  qHBoxLayout->addWidget(pushButtonReloadData);
  pushButtonShowHelp = new QPushButton;
  pushButtonShowHelp->setIcon(DIcon::Help());
  connect(pushButtonShowHelp, SIGNAL(clicked()), this, SLOT(helpSlot()));
  qHBoxLayout->addWidget(pushButtonShowHelp);
  qHBoxLayout->addStretch(1);
  buttonGroup->setLayout(qHBoxLayout);
  mainVLayout->addWidget(buttonGroup);
  QHBoxLayout *mainHLayout = new QHBoxLayout;
  qTextBrowser1 = new QTextBrowser;
  mainHLayout->addWidget(qTextBrowser1);
  qTextBrowser2 = new QTextBrowser;
  mainHLayout->addWidget(qTextBrowser2);
  mainVLayout->addLayout(mainHLayout);
  retranslateUI();
  widMain->setLayout(mainVLayout);
  setWidget(widMain);
  secondsToNextIterationTimer->start(1000);
}

void DMonitoring::retranslateUI()
{
  setWindowTitle(tr("Monitoring"));
  setObjectName(windowTitle());
  dTitleLabel->setText(windowTitle());
  pushButtonSettings->setText(tr("Settings"));
  pushButtonSettings->setToolTip(pushButtonSettings->text());
  spinBoxRefreshRate->setToolTip(tr("Refresh rate."));
  spinBoxRefreshRate->setPrefix(tr("Refresh rate:") + " ");
  spinBoxRefreshRate->setSuffix(" " + tr("seconds"));
  pushButtonStopRefreshing->setText(tr("Stop refreshing"));
  pushButtonStopRefreshing->setToolTip(pushButtonStopRefreshing->text());
  buttonGroup->setTitle(tr("Actions"));
  pushButtonReloadData->setText(tr("Reload data"));
  pushButtonReloadData->setToolTip(pushButtonReloadData->text());
  pushButtonShowHelp->setText(tr("Help"));
  pushButtonShowHelp->setToolTip(pushButtonShowHelp->text());
}

bool DMonitoring::compareResultField(QString resultField, QString comparition, QString defaultValue)
{
//  qDebug() << resultField << comparition << defaultValue;
  if (comparition == "None")
    return false;
  if (comparition == "<")
    return !(defaultValue.trimmed() > resultField.trimmed());
  if (comparition == ">")
    return !(defaultValue.trimmed() < resultField.trimmed());
  if (comparition == "=")
    return !(defaultValue.trimmed() == resultField.trimmed());
  if (comparition == "<>")
    return !(defaultValue.trimmed() != resultField.trimmed());
  return false;
}

void DMonitoring::helpSlot()
{
  QMessageBox::information(this, windowTitle(), tr("This is how Connection Monitoring works:"
"<br />* The Settings button establishes the parameters for monitoring a new connection."
"<br />* The refresh rate is in seconds."
"<br />* The seconds pending for the next iteration are shown in the label."
"<br />* The Stop Refresh button stops updating the data."
"<br />* The Refresh data button reloads the data as long as the Stop Refresh button is not pressed."
"<br />* The Help button displays this text."
"<br />"
"<br />Add a record in Connection Monitoring:"
"<br />* To add a new monitored connection you must press the Add a new monitored connection button."
"<br />* Then you must select the connection to use."
"<br />* The Column name must contain the name of the column that the query will return."
"<br />* The Query is the instruction to be executed on the database server, it is expected to return a single record and the name of the column."
"<br />* The Comparition is the operator to use to determinate if the returned value is aceptable."
"<br />* The Value is the to compare with."
"<br />* The Close button closes the window."
"<br />* The Help button displays this text."
"<br />* The Save button saves the data in the configuration file."));
}

void DMonitoring::reloadData()
{
  if (pushButtonStopRefreshing->isChecked()) {
    timerRefresh->stop();
  } else {
    QApplication::setOverrideCursor(Qt::WaitCursor);
    emit loadProgress(0);
    QString data1;
    QString data2;
    data1 += "<TABLE border='1' align='center'>";
    data1 += "  <TR>";
    data1 += "    <TH>" + tr("Connection") + "</TH>";
    data1 += "    <TH>" + tr("Column name") + "</TH>";
    data1 += "    <TH>" + tr("Value") + "</TH>";
    data1 += "    <TH>" + tr("Status") + "</TH>";
    data1 += "  </TR>";
    data2 = data1;
    QStringList connectionsToMonitor = settings.allKeys("Monitoring");
    int valueToRemove = connectionsToMonitor.indexOf("RefreshRate");
    if (valueToRemove >= 0)
        connectionsToMonitor.removeAt(valueToRemove);
    QStringList values;
    QString bgcolor;
    QString lastConnectionToMonitor;
    QList<QStringList>* result;
    QString resultField;
    unsigned int bgcolorCounter = 0;
    unsigned int counter = 0;
    QString monitoringData;
    foreach (QString connectionToMonitor, connectionsToMonitor) {
      monitoringData.clear();
      values = connectionToMonitor.split("$$$");
      if (lastConnectionToMonitor != values.at(0)) {
        bgcolorCounter++;
        lastConnectionToMonitor = values.at(0);
        QMap<QString, QVariant> params = settings.value("ServerConnections/" + values.at(0)).toMap();
        serverConnection->close();
        serverConnection->setConnectionParameters(params);
        if (params.value("ConnectionType").toString() == "--")
          serverConnection->setDBMSType(StaticFunctions::Undefined);
        if (params.value("ConnectionType")  == "MySQL")
          serverConnection->setDBMSType(StaticFunctions::MySQL);
        if (params.value("ConnectionType")  == "MariaDB")
          serverConnection->setDBMSType(StaticFunctions::MariaDB);
        serverConnection->open();
      }
      bgcolor = bgcolorCounter % 2 ? "LightGrey" : "White";
      monitoringData += "  <TR>";
      monitoringData += "    <TD style='padding:3px' bgcolor='" + bgcolor + "'>" + values.at(0) + "</TD>";
      monitoringData += "    <TD style='padding:3px' bgcolor='" + bgcolor + "'>" + values.at(1) + "</TD>";
      result = serverConnection->runQuery(settings.value("Monitoring/" + connectionToMonitor, "").toString(), true, false);
      int valueToCheck = result->at(0).indexOf(values.at(1));
      if (valueToCheck >= 0)
        resultField = result->at(1).at(valueToCheck);
      else
        resultField = "Error";
      monitoringData += "    <TD style='padding:3px' bgcolor='" + bgcolor + "' align='right'>" + resultField + "</TD>";
      monitoringData += "    <TD style='padding:3px' bgcolor='"
          + ((compareResultField(resultField, values.at(2), values.at(3))) ? "red" : bgcolor )
          + "' align='right'>" + values.at(3) + "</TD>";
      monitoringData += "  </TR>";
      counter++;
      emit statusBarMessage(values.at(0), QSystemTrayIcon::Information, 5);
      emit loadProgress((int) (counter * 100 / connectionsToMonitor.count()));
      if (counter <= connectionsToMonitor.count() / 2) {
        data1 += monitoringData;
      } else {
        data2 += monitoringData;
      }
    }
    data1 += "</TABLE>";
    data1 += "<P align='center'>" + tr("Iterations") + ": " + QString::number(++interationsCount) + "</P>";
    data1 += "";
    data2 += "</TABLE>";
    data2 += "<P align='center'>" + tr("Iterations") + ": " + QString::number(interationsCount) + "</P>";
    data2 += "";
    qTextBrowser1->setHtml(data1);
    qTextBrowser2->setHtml(data2);
  }
  emit loadProgress(100);
  QApplication::restoreOverrideCursor();
}

void DMonitoring::refreshRateSlot(const int value)
{
  timerRefresh->setInterval(1000 * value);
  settings.setValue("Monitoring/RefreshRate", value);
}

void DMonitoring::pushButtonSettingsSlot()
{
  pushButtonStopRefreshing->setChecked(true);
  monitoredConnection = new MonitoredConnection;
  connect(monitoredConnection, SIGNAL(reloadData()), this, SLOT(reloadData()));
  connect(monitoredConnection, SIGNAL(showHelp()), this, SLOT(helpSlot()));
  monitoredConnection->setModal(true);
  monitoredConnection->show();
}

void DMonitoring::secondsToNextIterationSlot()
{
  secondsToNextIteration->setText(tr("Seconds to next iteration: ") + QString::number(timerRefresh->remainingTime() / 1000));
}

MonitoredConnection::MonitoredConnection()
{
  setSizeGripEnabled(true);
  setWindowIcon(DIcon::DocumentNew());
  setWindowTitle(tr("Monitored Connections"));
  setObjectName(windowTitle());
  setAttribute(Qt::WA_DeleteOnClose);
  QVBoxLayout *mainVLayout = new QVBoxLayout;
  mainVLayout->addWidget(new DTitleLabel(windowTitle()));
  headers = new QList<QStringList>;
  QStringList connections = settings.allKeys("ServerConnections");
  int valueToRemove = connections.indexOf("StorePassword");
  if (valueToRemove >= 0)
    connections.removeAt(valueToRemove);
  valueToRemove = connections.indexOf("SortConnectionList");
  if (valueToRemove >= 0)
    connections.removeAt(valueToRemove);
  headers->append(QStringList() << tr("Connection") << (StaticFunctions::DelegateTypeEnum() + "(" + connections.join(",") + ")")  << "" << "Left" << "localhost");
  headers->append(QStringList() << tr("Column name") << "NoDelegate"  << "" << "Left" << "Column name");
  headers->append(QStringList() << tr("Query") << "NoDelegate"  << "" << "Left" << "SELECT NOW() AS `Column name`");
  headers->append(QStringList() << tr("Comparition") << (StaticFunctions::DelegateTypeEnum() + "(None,<,>,<>,=)")  << "" << "Center" << "None");
  headers->append(QStringList() << tr("Value") << "NoDelegate"  << "" << "Left" << "2021-12-07 14:13:18");
  dTableView = new DTableView(headers);
  mainVLayout->addWidget(dTableView);
  QDialogButtonBox *buttonBox = new QDialogButtonBox(QDialogButtonBox::Save | QDialogButtonBox::Close | QDialogButtonBox::Help);
  buttonBox->addButton(tr("Add a monitored connection"), QDialogButtonBox::AcceptRole);
  buttonBox->addButton(tr("Remove the current connection"), QDialogButtonBox::AcceptRole);
  buttonBox->addButton(tr("Clone the current connection"), QDialogButtonBox::AcceptRole);
  connect(buttonBox->button(QDialogButtonBox::Save), SIGNAL(clicked()), this, SLOT(saveMonitoredConnections()));
  connect(buttonBox->button(QDialogButtonBox::Close), SIGNAL(clicked()), this, SLOT(reject()));
  connect(buttonBox->button(QDialogButtonBox::Help), SIGNAL(clicked()), this, SLOT(helpSlot()));
  connect(buttonBox, SIGNAL(clicked(QAbstractButton*)), this, SLOT(actionRoleSlot(QAbstractButton*)));
  mainVLayout->addWidget(buttonBox);
  setLayout(mainVLayout);
  loadMonitoredConnections();
}

void MonitoredConnection::saveMonitoredConnections()
{
  QStringList connectionsToMonitor = settings.allKeys("Monitoring");
  int valueToRemove = connectionsToMonitor.indexOf("RefreshRate");
  if (valueToRemove >= 0)
    connectionsToMonitor.removeAt(valueToRemove);
  foreach (QString connectionToMonitor, connectionsToMonitor) {
    settings.remove(connectionToMonitor);
  }
  for (unsigned int counter = 0; counter < dTableView->rowCount(); counter++) {
    QList<QStandardItem *> row = dTableView->getRow(counter);
    settings.setValue("Monitoring/"
                        + row.at(0)->text()
                        + "$$$" + row.at(1)->text()
                        + "$$$" + row.at(3)->text()
                        + "$$$" + row.at(4)->text()
                      , row.at(2)->text());
  }
  loadMonitoredConnections();
}

void MonitoredConnection::actionRoleSlot(QAbstractButton *button)
{
  if (button->text() == tr("Add a monitored connection")) {
    if (dTableView->model()->rowCount() == 0) {
      QList<QStringList> *monitoredConnections = new QList<QStringList>();
      QStringList monitoredConnection;
      monitoredConnection.append("localhost");
      monitoredConnection.append("CurrentTime");
      monitoredConnection.append("SELECT NOW() AS `CurrentTime`");
      monitoredConnection.append("=");
      monitoredConnection.append("2021-12-07 14:13:18");
      monitoredConnections->append(monitoredConnection);
      dTableView->setModelData(monitoredConnections, false, 0, false);
    } else {
      dTableView->addRow();
    }
  }
  if (button->text() == tr("Remove the current connection")) {
    dTableView->itemModel->removeRow(dTableView->currentIndexItem().row());
  }
  if (button->text() == tr("Clone the current connection")) {
    dTableView->cloneLine(dTableView->currentIndexItem().row());
//    QList<QStandardItem *> row = dTableView->getRow(dTableView->currentIndexItem().row());
//    QList<QStandardItem *> items;
//    items.append(new QStandardItem(row.at(0)->text()));
//    items.append(new QStandardItem(row.at(1)->text()));
//    items.append(new QStandardItem(row.at(2)->text()));
//    items.append(new QStandardItem(row.at(3)->text()));
//    items.append(new QStandardItem(row.at(4)->text()));
//    dTableView->itemModel->appendRow(items);
  }
}

void MonitoredConnection::loadMonitoredConnections()
{
  QStringList connectionsToMonitor = settings.allKeys("Monitoring");
  int valueToRemove = connectionsToMonitor.indexOf("RefreshRate");
  if (valueToRemove >= 0)
    connectionsToMonitor.removeAt(valueToRemove);
  QStringList values;
  QList<QStringList> *monitoredConnections = new QList<QStringList>();
  QStringList monitoredConnection;
  foreach (QString connectionToMonitor, connectionsToMonitor) {
    values = connectionToMonitor.split("$$$");
    monitoredConnection.clear();
    monitoredConnection.append(values.at(0));
    monitoredConnection.append(values.at(1));
    monitoredConnection.append(settings.value("Monitoring/" + connectionToMonitor, "").toString());
    monitoredConnection.append(values.at(2));
    if (values.count() > 3) {
      monitoredConnection.append(values.at(3));
    } else {
      monitoredConnection.append("");
    }
    monitoredConnections->append(monitoredConnection);
  }
  dTableView->setModelData(monitoredConnections, false, 0, false);
}

void MonitoredConnection::helpSlot()
{
  emit showHelp();
}
