/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QVBoxLayout>
#include <QFile>
#include <QMessageBox>
#include <QGroupBox>
#include <QFormLayout>
#include <QLabel>
#include <QGroupBox>
#include <QGridLayout>
#include <QDialogButtonBox>
#include <QFileInfo>
#include <QAbstractButton>
#include <QCheckBox>
#include <QLineEdit>
#include <QCompleter>
#include <QStringListModel>
#include <QFrame>

#include "dfileselector.h"
#include "dloaddata.h"
#include "dtitlelabel.h"
#include "basetexteditor.h"
#include "dmessagelabel.h"

DLoadData::DLoadData(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
  setWindowIcon(DIcon::LoadData());
  QVBoxLayout *mainVLayout = new QVBoxLayout;
  mainVLayout->setContentsMargins(3, 0, 3, 0);
  dTitleLabel = new DTitleLabel;
  mainVLayout->addWidget(dTitleLabel);

  dMessageLabel = new DMessageLabel(MessageTypes::Information);
  mainVLayout->addWidget(dMessageLabel);

  dFileSelector = new DFileSelector(DFileSelectorContexts::LoadDataFile);
  dFileSelector->setFileName(settings.value("LoadData/LastLoadedFile", "").toString());
  connect(dFileSelector, SIGNAL(changed()), this, SLOT(getFileHeader()));
  QFormLayout *fileLayout = new QFormLayout;
  fileLayout->addRow(dFileSelector);

  databaseCompleter = new QCompleter(this->serverConnection->getDatabases(true));
  databaseCompleter->setCaseSensitivity(Qt::CaseInsensitive);
  tableCompleter = new QCompleter();
  tableCompleter->setCaseSensitivity(Qt::CaseInsensitive);

  databaseToUse = new QLabel;
  tableToUse = new QLabel;
  databaseLineEdit = new QLineEdit;
  connect(databaseLineEdit, SIGNAL(textChanged(QString)), this, SLOT(databaseCompleterTextChanged(QString)));
  databaseLineEdit->setClearButtonEnabled(true);
  databaseLineEdit->setCompleter(databaseCompleter);
  databaseLineEdit->setText(settings.value("LoadData/Database", "").toString());

  tableLineEdit = new QLineEdit;
  tableLineEdit->setClearButtonEnabled(true);
  tableLineEdit->setCompleter(tableCompleter);
  tableLineEdit->setText(settings.value("LoadData/Table", "").toString());

  QHBoxLayout *hBoxLayout1 = new QHBoxLayout;
  hBoxLayout1->addWidget(databaseToUse);
  hBoxLayout1->addWidget(databaseLineEdit);
  hBoxLayout1->addWidget(tableToUse);
  hBoxLayout1->addWidget(tableLineEdit);
  fileLayout->addRow(hBoxLayout1);
  QFrame* sepatatorFrame = new QFrame();
  sepatatorFrame->setFrameShape(QFrame::HLine);
  fileLayout->addWidget(sepatatorFrame);

  fileGroupBox = new QGroupBox;
  fileGroupBoxLayout = new QFormLayout;
  fileLabel = new QLabel;
  fileLabel->setTextFormat(Qt::PlainText);
  dataLengthLabel = new QLabel;
  fileGroupBoxLayout->addRow(" ", fileLabel);
  fileGroupBoxLayout->addRow(" ", dataLengthLabel);
  showTop100Rows = new QCheckBox;
  showTop100Rows->setChecked(settings.value("LoadData/ShowTop100Rows", false).toBool());
  showTop100Rows->setAttribute(Qt::WA_AlwaysShowToolTips, true);
  connect(showTop100Rows, SIGNAL(stateChanged(int)), this, SLOT(getFileHeader(int)));
  useFirstLineForHeaders = new QCheckBox;
  useFirstLineForHeaders->setChecked(settings.value("LoadData/UseFirstLineForHeaders", false).toBool());
  useFirstLineForHeaders->setAttribute(Qt::WA_AlwaysShowToolTips, true);
  connect(useFirstLineForHeaders, SIGNAL(stateChanged(int)), this, SLOT(getFileHeader(int)));
  createABasicTable = new QCheckBox;
  createABasicTable->setChecked(settings.value("LoadData/CreateABasicTable", false).toBool());
  createABasicTable->setAttribute(Qt::WA_AlwaysShowToolTips, true);
  connect(createABasicTable, SIGNAL(stateChanged(int)), this, SLOT(getFileHeader(int)));
  optionsGroupBox = new QGroupBox;
  baseTextEditor = new BaseTextEditor(EditorTypes::NoEditor);
  baseTextEditor->setWordWrapMode(QTextOption::WordWrap);
  baseTextEditor->showHideLineNumbers(false);
  baseTextEditor->setWordWrapMode(QTextOption::NoWrap);
  mainVLayout->addLayout(fileLayout);

  fileGroupBox->setLayout(fileGroupBoxLayout);
  mainVLayout->addWidget(fileGroupBox);

  optionsGroupBoxLayout = new QGridLayout;
  optionsGroupBoxLayout->addWidget(showTop100Rows, 0, 0, Qt::AlignLeft);
  optionsGroupBoxLayout->addWidget(useFirstLineForHeaders, 0, 1, Qt::AlignLeft);
  optionsGroupBoxLayout->addWidget(createABasicTable, 0, 2, Qt::AlignLeft);

  optionsGroupBox->setLayout(optionsGroupBoxLayout);
  mainVLayout->addWidget(optionsGroupBox);
  mainVLayout->addWidget(baseTextEditor);

  buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Help);
  connect(buttonBox, SIGNAL(accepted()), this, SLOT(acceptedSlot()));
  mainVLayout->addWidget(buttonBox);

  mainVLayout->addLayout(fileLayout);
  retranslateUI();
  QWidget *widMain = new QWidget;
  widMain->setLayout(mainVLayout);
  setWidget(widMain);
  getFileHeader();
}

void DLoadData::retranslateUI()
{
  setWindowTitle(tr("Load Data"));
  setObjectName(windowTitle());
  dTitleLabel->setText(windowTitle());
  dTitleLabel->setToolTip(dTitleLabel->text());
  dFileSelector->setText(tr("&File:"));
  fileGroupBox->setTitle(tr("File information"));
  QLabel *label = qobject_cast<QLabel *>(fileGroupBoxLayout->labelForField(fileLabel));
  label->setText(tr("File:"));
  label->setToolTip(label->text());
  label = qobject_cast<QLabel *>(fileGroupBoxLayout->labelForField(dataLengthLabel));
  label->setText(tr("Total data length:"));
  label->setToolTip(label->text());
  optionsGroupBox->setTitle(tr("Options"));
  optionsGroupBox->setToolTip(optionsGroupBox->title());
  showTop100Rows->setText(tr("Show top 100 rows"));
  showTop100Rows->setStatusTip(showTop100Rows->toolTip());
  showTop100Rows->setWhatsThis(showTop100Rows->toolTip());
  useFirstLineForHeaders->setText(tr("Use first line for headers"));
  useFirstLineForHeaders->setStatusTip(useFirstLineForHeaders->toolTip());
  useFirstLineForHeaders->setWhatsThis(useFirstLineForHeaders->toolTip());
  createABasicTable->setText(tr("Create a basic table"));
  createABasicTable->setStatusTip(createABasicTable->toolTip());
  createABasicTable->setWhatsThis(createABasicTable->toolTip());
  dMessageLabel->setText(tr("Load Data uses LOAD DATA LOCAL INFILE to load the data."
                            "Please have in mind the specifics and the limitation of using it."
                            "<br />LOAD DATA INFILE is unsafe for statement-based replication."
                            "<br />The given path must have the rigth read privileges."));
  tableToUse->setText(tr("Table to use:"));
  databaseToUse->setText(tr("Database to use:"));
  databaseLineEdit->setPlaceholderText(tr("Database"));
  tableLineEdit->setPlaceholderText(tr("Table"));
}

QString DLoadData::loadDataStatement()
{
  return "LOAD DATA LOCAL INFILE '" + dFileSelector->getFileName()
      + "' INTO TABLE `" + databaseLineEdit->text() + "`.`" + tableLineEdit->text()
      + "` FIELDS TERMINATED BY ','"
      + (useFirstLineForHeaders->isChecked() ? " IGNORE 1 LINES" : "")
      + "\n\n";
}

void DLoadData::getFileHeader(int state)
{
  Q_UNUSED(state);
  QFileInfo fileInfo(dFileSelector->getFileName());
  fileLabel->setText(dFileSelector->getFileName());
  fileLabel->setToolTip(fileLabel->text());
  dataLengthLabel->setText("~" + StaticFunctions::bytesConvertor(QString("%1").arg(fileInfo.size())));
  dataLengthLabel->setToolTip(dataLengthLabel->text());

  QString output;
  QList<QStringList> *rows = new QList<QStringList>();
  QList<int> maxWidthList;
  int row = 0;
  int row2 = 0;
  settings.setValue("LoadData/LastLoadedFile", dFileSelector->getFileName());
  settings.setValue("LoadData/UseFirstLineForHeaders", useFirstLineForHeaders->isChecked());
  settings.setValue("LoadData/CreateABasicTable", createABasicTable->isChecked());
  settings.setValue("LoadData/ShowTop100Rows", showTop100Rows->isChecked());
  settings.setValue("LoadData/Database", databaseLineEdit->text());
  settings.setValue("LoadData/Table", tableLineEdit->text());

  output = tr("This is a preview of the statement to execute:") + "\n" + loadDataStatement();

  if (createABasicTable->isChecked())
    useFirstLineForHeaders->setChecked(true);

  if (showTop100Rows->isChecked() || createABasicTable->isChecked()) {
    QFile fileToOpen(dFileSelector->getFileName());
    bool opened = fileToOpen.open(QFile::ReadOnly);
    Q_UNUSED(opened);
    if (!fileToOpen.isOpen()) {
      QMessageBox::information(this, tr("File can not be opened"), dFileSelector->getFileName() + "<br />");
      return;
    }
    QTextStream stream(&fileToOpen);
    unsigned int counter = 0;
    for (QString line = stream.readLine(); !line.isNull() && counter < 100; line = stream.readLine()) {
      rows->append(line.split(","));
      counter++;
    }

    //Find the maximum values for the width of the columns
    for (row = 0; row < rows->at(0).count(); row++)
      maxWidthList.append(1);
    for (row = 0; row < (rows->count() - 1); row++)
      for (row2 = 0; row2 < rows->at(row).count(); row2++)
        if (QString(rows->at(row).at(row2)).length() > maxWidthList.at(row2))
          maxWidthList.replace(row2, QString(rows->at(row).at(row2)).length());
  }

  if (createABasicTable->isChecked() && !rows->isEmpty()) {
    output += tr("This is a preview of the table to create:") + "\n";
    output += "DROP TABLE IF EXISTS `" + databaseLineEdit->text() + "`.`" + tableLineEdit->text() + "`;\n";
    output += "CREATE TABLE `" + databaseLineEdit->text() + "`.`" + tableLineEdit->text() + "` (\n";
    for (row = 0; row < rows->at(0).count(); row++) {
      output += QString("`%1` TEXT, \n").arg(rows->at(0).at(row));
    }
    output = output.left(output.size() - 3);
    output += ");\n\n";
  }
  if (showTop100Rows->isChecked() && !rows->isEmpty()) {
    output += tr("This is a preview of the data to load:") + "\n";
    //Prints the headres and theirs borders
    if (useFirstLineForHeaders->isChecked()) {
      output += "+";
      for (row = 0; row < rows->at(0).count(); row++)
        output +=  QString('-').repeated(maxWidthList.at(row) + 2) + "+";
      output += "\n|";
      for (row = 0; row < rows->at(0).count(); row++)
        output +=  " " + rows->at(0).at(row).leftJustified(maxWidthList.at(row), ' ') + (((row + 1) == rows->at(0).count()) ? "" : " |");
      output += " |\n";
    }
    output += "+";
    for (row = 0; row < rows->at(0).count(); row++)
      output +=  QString('-').repeated(maxWidthList.at(row) + 2) + "+";

    //Print the data
    for (row = (useFirstLineForHeaders->isChecked() ? 1 : 0); row < rows->count() - 1; row++) {
      output += "\n|";
      for (row2 = 0; row2 < rows->at(row).count(); row2++)
        output +=  " " + rows->at(row).at(row2).toUtf8().leftJustified(maxWidthList.at(row2), ' ') + (((row2 + 1) == rows->at(0).count()) ? "" : " |");
      output += " |";
    }

    //Print the botton border
    output += "\n+";
    for (row = 0; row < rows->at(0).count(); row++)
      output +=  QString('-').repeated(maxWidthList.at(row) + 2) + "+";
  }

  delete(rows);
  baseTextEditor->setPlainText(output);
}

void DLoadData::acceptedSlot()
{
  baseTextEditor->clear();
  QString output;
  if (createABasicTable->isChecked()) {
    useFirstLineForHeaders->setChecked(true);

    QFile fileToOpen(dFileSelector->getFileName());
    bool opened = fileToOpen.open(QFile::ReadOnly);
    Q_UNUSED(opened);
    if (!fileToOpen.isOpen()) {
      QMessageBox::information(this, tr("File can not be opened"), dFileSelector->getFileName() + "<br />");
      return;
    }
    QTextStream stream(&fileToOpen);
    output += "CREATE TABLE `" + databaseLineEdit->text() + "`.`" + tableLineEdit->text() + "` (\n";
    foreach(QString column, stream.readLine().split(",")) {
      output += QString("`%1` TEXT, \n").arg(column);
    }
    output = output.left(output.size() - 3);
    output += ");";
    baseTextEditor->appendPlainText(this->serverConnection->runQuery("DROP TABLE IF EXISTS `" + databaseLineEdit->text() + "`.`" + tableLineEdit->text() + "`;", true)->at(0).at(0));
    baseTextEditor->appendPlainText(this->serverConnection->runQuery(output, true)->at(0).at(0));
  }
  baseTextEditor->appendPlainText(this->serverConnection->runQuery(loadDataStatement(), true)->at(0).at(0));
}

void DLoadData::databaseCompleterTextChanged(QString text)
{
  tableCompleter->setModel(new QStringListModel(this->serverConnection->database(text)->getLocalTables(), tableCompleter));
}
