/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DLINEEDIT_H
#define DLINEEDIT_H

#include <QLineEdit>
#include <QIcon>

class QToolButton;

/**
 * @brief The DLineEdit class is a customized QLineEdit with an embedded action button.
 *
 * This widget extends the standard single-line text editor to include a clickable
 * icon (QToolButton) inside the text area. This is commonly used for search boxes
 * (search icon) or inputs with a clear button (X icon).
 */
class DLineEdit : public QLineEdit
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the DLineEdit widget.
   * @param icon The icon to display within the action button.
   * @param autoHideIcon If true, the button visibility toggles based on text content; otherwise, it remains visible.
   */
  DLineEdit(QIcon icon, bool autoHideIcon = false);

private:
  QIcon icon;
  QToolButton *actionButton;

protected:
  /**
   * @brief Handles resize events to adjust the action button position.
   * @param event The resize event parameters.
   */
  void resizeEvent(QResizeEvent *event);

  /**
   * @brief Handles mouse press events to trigger the clicked signal.
   * @param event The mouse event parameters.
   */
  void mousePressEvent(QMouseEvent *event);

private slots:
  /**
   * @brief Toggles the action button's visibility based on the text.
   * @param text The current text in the line edit.
   */
  void showHideActionButton(QString text);

  /**
   * @brief Internal slot to forward the text changed signal.
   */
  void emitTextChangedSignal();

  /**
   * @brief Internal slot to handle the return key press.
   */
  void returnPressedSlot();

signals:
  /**
   * @brief Emitted whenever the text changes.
   * @param text The new text string.
   */
  void textChangedSignal(const QString &text);

  /**
   * @brief Emitted when the widget receives a mouse press event.
   */
  void clicked();

  /**
   * @brief Emitted when the Return or Enter key is pressed.
   * @param text The text contained in the widget at the moment of the key press.
   */
  void returnKeyPressed(const QString &text);
};

#endif // DLINEEDIT_H
