/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DITEMDELEGATE_H
#define DITEMDELEGATE_H

#include <QItemDelegate>

/**
 * @brief The DItemDelegate class provides custom editing widgets for item views.
 *
 * This class extends QItemDelegate to support specific data types (passed via
 * the constructor). It is particularly useful for handling database fields like
 * ENUMs or Booleans by providing specialized editors (e.g., QComboBox) instead
 * of the default QLineEdit.
 */
class DItemDelegate : public QItemDelegate
{
public:
  /**
   * @brief Constructs the DItemDelegate.
   * @param delegateType A string identifier indicating the type of editor to create (e.g., "ENUM", "SET").
   */
  DItemDelegate(QString delegateType);

  /**
   * @brief Creates the editor widget for the specific item.
   *
   * Overrides QAbstractItemDelegate::createEditor to return a specific widget
   * based on the `delegateType` (e.g., a ComboBox for enums).
   *
   * @param parent The parent widget for the editor.
   * @param option The style options for the item.
   * @param index The model index being edited.
   * @return Pointer to the newly created editor widget.
   */
  QWidget *createEditor(QWidget *parent, const QStyleOptionViewItem &option, const QModelIndex &index) const;

  /**
   * @brief Populates the editor with data from the model.
   *
   * Overrides QAbstractItemDelegate::setEditorData.
   *
   * @param editor Pointer to the editor widget.
   * @param index The model index containing the current data.
   */
  void setEditorData(QWidget *editor, const QModelIndex &index) const;

  /**
   * @brief Saves data from the editor back to the model.
   *
   * Overrides QAbstractItemDelegate::setModelData.
   *
   * @param editor Pointer to the editor widget.
   * @param model Pointer to the abstract item model.
   * @param index The model index where data should be saved.
   */
  void setModelData(QWidget *editor, QAbstractItemModel *model, const QModelIndex &index) const;

  /**
   * @brief Updates the geometry of the editor to fit the item view.
   *
   * Overrides QAbstractItemDelegate::updateEditorGeometry to ensure the editor
   * is displayed correctly over the cell.
   *
   * @param editor Pointer to the editor widget.
   * @param option The style options containing the target rectangle.
   * @param index The model index being edited.
   */
  void updateEditorGeometry(QWidget *editor, const QStyleOptionViewItem &option, const QModelIndex &index) const;

private:
  QString delegateType;
  QStringList enumOptions;
};

#endif // DITEMDELEGATE_H
