/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DICON_H
#define DICON_H

#include <QObject>
#include <QIcon>
#include <QMovie>

/**
 * @brief The DIcon class acts as a central repository for application icons.
 *
 * This static helper class provides convenient access to standard QIcon objects
 * and their corresponding resource paths used throughout the application.
 */
class DIcon : public QIcon
{
public:
  DIcon();

  /**
   * @brief Retrieves the icon representing a new document.
   * @return The 'New Document' QIcon.
   */
  static QIcon DocumentNew();

  /**
   * @brief Retrieves the resource path for the new document icon.
   * @return A QString containing the file path.
   */
  static QString DocumentNewIconPath();

  /**
   * @brief Retrieves the icon for closing a document.
   * @return The 'Close Document' QIcon.
   */
  static QIcon DocumentClose();

  /**
   * @brief Retrieves the resource path for the close document icon.
   * @return A QString containing the file path.
   */
  static QString DocumentCloseIconPath();

  /**
   * @brief Retrieves the icon for import operations.
   * @return The 'Import' QIcon.
   */
  static QIcon Import();

  /**
   * @brief Retrieves the resource path for the import icon.
   * @return A QString containing the file path.
   */
  static QString ImportIconPath();

  /**
   * @brief Retrieves the icon for printing.
   * @return The 'Print' QIcon.
   */
  static QIcon Print();

  /**
   * @brief Retrieves the icon for opening a document.
   * @return The 'Open' QIcon.
   */
  static QIcon DocumentOpen();

  /**
   * @brief Retrieves the resource path for the open document icon.
   * @return A QString containing the file path.
   */
  static QString DocumentOpenIconPath();

  /**
   * @brief Retrieves the icon for saving.
   * @return The 'Save' QIcon.
   */
  static QIcon Save();

  /**
   * @brief Retrieves the icon for saving as a new file.
   * @return The 'Save As' QIcon.
   */
  static QIcon SaveAs();

  /**
   * @brief Retrieves the icon for the copy action.
   * @return The 'Copy' QIcon.
   */
  static QIcon Copy();

  /**
   * @brief Retrieves the icon for the cut action.
   * @return The 'Cut' QIcon.
   */
  static QIcon Cut();

  /**
   * @brief Retrieves the icon for the paste action.
   * @return The 'Paste' QIcon.
   */
  static QIcon Paste();

  /**
   * @brief Retrieves the icon for the undo action.
   * @return The 'Undo' QIcon.
   */
  static QIcon Undo();

  /**
   * @brief Retrieves the icon for the redo action.
   * @return The 'Redo' QIcon.
   */
  static QIcon Redo();

  /**
   * @brief Retrieves the icon for clearing text or content.
   * @return The 'Clear' QIcon.
   */
  static QIcon Clear();

  /**
   * @brief Retrieves the resource path for the clear icon.
   * @return A QString containing the file path.
   */
  static QString ClearIconPath();

  /**
   * @brief Retrieves the icon for reverting a document.
   * @return The 'Revert' QIcon.
   */
  static QIcon DocumentRevert();

  /**
   * @brief Retrieves the resource path for the revert document icon.
   * @return A QString containing the file path.
   */
  static QString DocumentRevertIconPath();

  /**
   * @brief Retrieves the icon for zooming in.
   * @return The 'Zoom In' QIcon.
   */
  static QIcon ZoomIn();

  /**
   * @brief Retrieves the icon for zooming out.
   * @return The 'Zoom Out' QIcon.
   */
  static QIcon ZoomOut();

  /**
   * @brief Retrieves the icon for resetting zoom to original size.
   * @return The 'Zoom Original' QIcon.
   */
  static QIcon ZoomOriginal();

  /**
   * @brief Retrieves the icon for jumping to a specific location (Go To).
   * @return The 'Go To' QIcon.
   */
  static QIcon GoTo();

  /**
   * @brief Retrieves the icon for selecting all items.
   * @return The 'Select All' QIcon.
   */
  static QIcon SelectAll();

  /**
   * @brief Retrieves the icon for refreshing data.
   * @return The 'Refresh' QIcon.
   */
  static QIcon Refresh();

  /**
   * @brief Retrieves the icon for adding an item to a list.
   * @return The 'List Add' QIcon.
   */
  static QIcon ListAdd();

  /**
   * @brief Retrieves the icon for removing an item from a list.
   * @return The 'List Remove' QIcon.
   */
  static QIcon ListRemove();

  /**
   * @brief Retrieves the icon for database rollback operations.
   * @return The 'Rollback' QIcon.
   */
  static QIcon DatabaseRollback();

  /**
   * @brief Retrieves the resource path for the database rollback icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseRollbackIconPath();

  /**
   * @brief Retrieves the generic database icon.
   * @return The 'Database' QIcon.
   */
  static QIcon Database();

  /**
   * @brief Retrieves the resource path for the generic database icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseIconPath();

  /**
   * @brief Retrieves the icon representing a database table.
   * @return The 'Table' QIcon.
   */
  static QIcon DatabaseTable();

  /**
   * @brief Retrieves the resource path for the database table icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseTableIconPath();

  /**
   * @brief Retrieves the icon for table status.
   * @return The 'Table Status' QIcon.
   */
  static QIcon DatabaseTableStatus();

  /**
   * @brief Retrieves the icon for a database event.
   * @return The 'Event' QIcon.
   */
  static QIcon DatabaseEvent();

  /**
   * @brief Retrieves the resource path for the database event icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseEventIconPath();

  /**
   * @brief Retrieves the icon for a database view.
   * @return The 'View' QIcon.
   */
  static QIcon DatabaseView();

  /**
   * @brief Retrieves the resource path for the database view icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseViewIconPath();

  /**
   * @brief Retrieves the icon for a database server.
   * @return The 'Server' QIcon.
   */
  static QIcon DatabaseServer();

  /**
   * @brief Retrieves the icon for a database user.
   * @return The 'User' QIcon.
   */
  static QIcon DatabaseUser();

  /**
   * @brief Retrieves the icon for opening recent documents.
   * @return The 'Open Recent' QIcon.
   */
  static QIcon DocumentOpenRecent();

  /**
   * @brief Retrieves the resource path for the open recent icon.
   * @return A QString containing the file path.
   */
  static QString DocumentOpenRecentIconPath();

  /**
   * @brief Retrieves the icon for executable files.
   * @return The 'Executable' QIcon.
   */
  static QIcon Executable();

  /**
   * @brief Retrieves the icon for PDF files.
   * @return The 'PDF' QIcon.
   */
  static QIcon PDF();

  /**
   * @brief Retrieves the icon for SVG images.
   * @return The 'SVG' QIcon.
   */
  static QIcon ImageSVG();

  /**
   * @brief Retrieves the icon for PNG images.
   * @return The 'PNG' QIcon.
   */
  static QIcon ImagePNG();

  /**
   * @brief Retrieves the icon for closing a view.
   * @return The 'View Close' QIcon.
   */
  static QIcon ViewClose();

  /**
   * @brief Retrieves the icon for splitting a view vertically (Top/Bottom).
   * @return The 'Split Top/Bottom' QIcon.
   */
  static QIcon ViewSplitTopBottom();

  /**
   * @brief Retrieves the icon for splitting a view horizontally (Left/Right).
   * @return The 'Split Left/Right' QIcon.
   */
  static QIcon ViewSplitLeftRight();

  /**
   * @brief Retrieves the icon for filtering views.
   * @return The 'Filter' QIcon.
   */
  static QIcon ViewFilter();

  /**
   * @brief Retrieves the icon for sorting in ascending order.
   * @return The 'Sort Ascending' QIcon.
   */
  static QIcon ViewSortAscending();

  /**
   * @brief Retrieves the icon for fullscreen mode.
   * @return The 'Fullscreen' QIcon.
   */
  static QIcon ViewFullscreen();

  /**
   * @brief Retrieves the icon for increasing indentation.
   * @return The 'Indent More' QIcon.
   */
  static QIcon FormatIndentMore();

  /**
   * @brief Retrieves the icon for decreasing indentation.
   * @return The 'Indent Less' QIcon.
   */
  static QIcon FormatIndentLess();

  /**
   * @brief Retrieves the icon for export operations.
   * @return The 'Export' QIcon.
   */
  static QIcon Export();

  /**
   * @brief Retrieves the icon for query execution.
   * @return The 'Execute Query' QIcon.
   */
  static QIcon QueryExecution();

  /**
   * @brief Retrieves the resource path for the query execution icon.
   * @return A QString containing the file path.
   */
  static QString QueryExecutionIconPath();

  /**
   * @brief Retrieves the icon for stopping a process.
   * @return The 'Stop' QIcon.
   */
  static QIcon Stop();

  /**
   * @brief Retrieves the standard folder icon.
   * @return The 'Folder' QIcon.
   */
  static QIcon Folder();

  /**
   * @brief Retrieves the resource path for the folder icon.
   * @return A QString containing the file path.
   */
  static QString FolderIconPath();

  /**
   * @brief Retrieves the icon for the database query log.
   * @return The 'Query Log' QIcon.
   */
  static QIcon DatabaseQueryLog();

  /**
   * @brief Retrieves the icon for database replication.
   * @return The 'Replication' QIcon.
   */
  static QIcon DatabaseReplication();

  /**
   * @brief Retrieves the icon for database maintenance.
   * @return The 'Maintenance' QIcon.
   */
  static QIcon DatabaseMaintenance();

  /**
   * @brief Retrieves the icon for database query history.
   * @return The 'History' QIcon.
   */
  static QIcon DatabaseQueryHistory();

  /**
   * @brief Retrieves the icon for information messages.
   * @return The 'Information' QIcon.
   */
  static QIcon Information();

  /**
   * @brief Retrieves the icon for warning dialogs.
   * @return The 'Warning' QIcon.
   */
  static QIcon DialogWarning();

  /**
   * @brief Retrieves the resource path for the warning dialog icon.
   * @return A QString containing the file path.
   */
  static QString DialogWarningIconPath();

  /**
   * @brief Retrieves the icon for error messages.
   * @return The 'Error' QIcon.
   */
  static QIcon Error();

  /**
   * @brief Retrieves the icon for the close action.
   * @return The 'Close' QIcon.
   */
  static QIcon Close();

  /**
   * @brief Retrieves the icon for the text editor.
   * @return The 'Text Editor' QIcon.
   */
  static QIcon TextEditor();

  /**
   * @brief Retrieves the icon for charts and graphics.
   * @return The 'Charts' QIcon.
   */
  static QIcon GraphicsCharts();

  /**
   * @brief Retrieves the icon for delete operations.
   * @return The 'Delete' QIcon.
   */
  static QIcon Delete();

  /**
   * @brief Retrieves the resource path for the delete icon.
   * @return A QString containing the file path.
   */
  static QString DeleteIconPath();

  /**
   * @brief Retrieves the icon for the trash can.
   * @return The 'Trash' QIcon.
   */
  static QIcon Trash();

  /**
   * @brief Retrieves the icon for a full trash can.
   * @return The 'Trash Full' QIcon.
   */
  static QIcon TrashFull();

  /**
   * @brief Retrieves the icon for a locked object.
   * @return The 'Locked' QIcon.
   */
  static QIcon ObjectLocked();

  /**
   * @brief Retrieves the icon for an unlocked object.
   * @return The 'Unlocked' QIcon.
   */
  static QIcon ObjectUnlocked();

  /**
   * @brief Retrieves the icon for character set settings.
   * @return The 'Character Set' QIcon.
   */
  static QIcon CharacterSet();

  /**
   * @brief Retrieves the icon for the find action.
   * @return The 'Find' QIcon.
   */
  static QIcon Find();

  /**
   * @brief Retrieves the icon for the replace action.
   * @return The 'Replace' QIcon.
   */
  static QIcon Replace();

  /**
   * @brief Retrieves the Qt logo icon.
   * @return The 'Qt' QIcon.
   */
  static QIcon QtLogo();

  /**
   * @brief Retrieves the MariaDB seal icon.
   * @return The 'MariaDB' QIcon.
   */
  static QIcon MariaDBSeal();

  /**
   * @brief Retrieves the SVN logo icon.
   * @return The 'SVN' QIcon.
   */
  static QIcon SVNLogo();

//  static QIcon ExecutableScript();

  /**
   * @brief Retrieves the icon for opening a wallet.
   * @return The 'Wallet Open' QIcon.
   */
  static QIcon WalletOpen();

  /**
   * @brief Retrieves the icon for starting media playback.
   * @return The 'Play' QIcon.
   */
  static QIcon MediaPlaybackStart();

  /**
   * @brief Retrieves the resource path for the media play icon.
   * @return A QString containing the file path.
   */
  static QString MediaPlaybackStartIconPath();

  /**
   * @brief Retrieves the icon for seeking backward in media.
   * @return The 'Seek Backward' QIcon.
   */
  static QIcon MediaSeekBackward();

  /**
   * @brief Retrieves the resource path for the seek backward icon.
   * @return A QString containing the file path.
   */
  static QString MediaSeekBackwardIconPath();

  /**
   * @brief Retrieves the icon for stopping media playback.
   * @return The 'Stop' QIcon.
   */
  static QIcon MediaPlaybackStop();

  /**
   * @brief Retrieves the resource path for the media stop icon.
   * @return A QString containing the file path.
   */
  static QString MediaPlaybackStopIconPath();

  /**
   * @brief Retrieves the icon for seeking forward in media.
   * @return The 'Seek Forward' QIcon.
   */
  static QIcon MediaSeekForward();

  /**
   * @brief Retrieves the resource path for the seek forward icon.
   * @return A QString containing the file path.
   */
  static QString MediaSeekForwardIconPath();

  /**
   * @brief Retrieves the icon for repeating query execution.
   * @return The 'Repeat Execution' QIcon.
   */
  static QIcon RepeateQueryExecution();

  /**
   * @brief Retrieves the icon for skipping forward.
   * @return The 'Skip Forward' QIcon.
   */
  static QIcon MediaSkipForward();

  /**
   * @brief Retrieves the resource path for the skip forward icon.
   * @return A QString containing the file path.
   */
  static QString MediaSkipForwardIconPath();

  /**
   * @brief Retrieves the icon for skipping backward.
   * @return The 'Skip Backward' QIcon.
   */
  static QIcon MediaSkipBackward();

  /**
   * @brief Retrieves the resource path for the skip backward icon.
   * @return A QString containing the file path.
   */
  static QString MediaSkipBackwardIconPath();

  /**
   * @brief Retrieves the icon for Hard Disk Drive usage.
   * @return The 'HDD Usage' QIcon.
   */
  static QIcon HDDUsage();

  /**
   * @brief Retrieves the icon for attaching a result.
   * @return The 'Attach Result' QIcon.
   */
  static QIcon AttachResult();

  /**
   * @brief Retrieves the icon for pausing media.
   * @return The 'Pause' QIcon.
   */
  static QIcon MediaPause();

  /**
   * @brief Retrieves the icon for locking query execution.
   * @return The 'Lock Query' QIcon.
   */
  static QIcon LockQueryExecution();

  /**
   * @brief Retrieves the icon for closing a window.
   * @return The 'Window Close' QIcon.
   */
  static QIcon WindowClose();

  /**
   * @brief Retrieves the resource path for the window close icon.
   * @return A QString containing the file path.
   */
  static QString WindowCloseIconPath();

  /**
   * @brief Retrieves the icon for system shutdown.
   * @return The 'Shutdown' QIcon.
   */
  static QIcon SystemShutdown();

  /**
   * @brief Retrieves the icon for system reboot.
   * @return The 'Reboot' QIcon.
   */
  static QIcon Reboot();

  /**
   * @brief Retrieves the icon for taking a screenshot.
   * @return The 'Screenshot' QIcon.
   */
  static QIcon Screenshot();

  /**
   * @brief Retrieves the icon for the character map accessory.
   * @return The 'Character Map' QIcon.
   */
  static QIcon AccessoriesCharacterMap();

  /**
   * @brief Retrieves the flag icon for the United States.
   * @return The 'US Flag' QIcon.
   */
  static QIcon FlagUS();

  /**
   * @brief Retrieves the flag icon for Costa Rica.
   * @return The 'CR Flag' QIcon.
   */
  static QIcon FlagCR();

  /**
   * @brief Retrieves the generic help icon.
   * @return The 'Help' QIcon.
   */
  static QIcon Help();

  /**
   * @brief Retrieves the GitLab logo icon.
   * @return The 'GitLab' QIcon.
   */
  static QIcon GitLab();

  /**
   * @brief Retrieves the icon representing downloads.
   * @return The 'Downloads' QIcon.
   */
  static QIcon EmblemDownloads();

  /**
   * @brief Retrieves the resource path for the downloads icon.
   * @return A QString containing the file path.
   */
  static QString EmblemDownloadsIconPath();

  /**
   * @brief Retrieves a movie object for a loading animation.
   * @return A QMovie representing the loading state.
   */
  static QMovie Loading();

  /**
   * @brief Retrieves the path to a random Calíope branding image at 512px.
   * @return A QString containing the file path.
   */
  static QString RandomCaliopeAt512px();

  /**
   * @brief Retrieves the icon for a database routine.
   * @return The 'Routine' QIcon.
   */
  static QIcon DatabaseRoutine();

  /**
   * @brief Retrieves the resource path for the database routine icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseRoutineIconPath();

  /**
   * @brief Retrieves the icon for a database trigger.
   * @return The 'Trigger' QIcon.
   */
  static QIcon DatabaseTrigger();

  /**
   * @brief Retrieves the resource path for the database trigger icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseTriggerIconPath();

  /**
   * @brief Retrieves the icon for a database query.
   * @return The 'Query' QIcon.
   */
  static QIcon DatabaseQuery();

  /**
   * @brief Retrieves the icon for the database process list.
   * @return The 'Process List' QIcon.
   */
  static QIcon DatabaseProcessList();

  /**
   * @brief Retrieves the Calíope application logo.
   * @return The 'Calíope' QIcon.
   */
  static QIcon Caliope();

  /**
   * @brief Retrieves the dashboard icon.
   * @return The 'Dashboard' QIcon.
   */
  static QIcon Dashboard();

  /**
   * @brief Retrieves the resource path for the dashboard icon.
   * @return A QString containing the file path.
   */
  static QString DashboardIconPath();

  /**
   * @brief Retrieves the preferences icon.
   * @return The 'Preferences' QIcon.
   */
  static QIcon Preferences();

  /**
   * @brief Retrieves the resource path for the preferences icon.
   * @return A QString containing the file path.
   */
  static QString PreferencesIconPath();

  /**
   * @brief Retrieves the icon for database backups.
   * @return The 'Backup' QIcon.
   */
  static QIcon DatabaseBackup();

  /**
   * @brief Retrieves the resource path for the database backup icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseBackupIconPath();

  /**
   * @brief Retrieves the icon for database restore operations.
   * @return The 'Restore' QIcon.
   */
  static QIcon DatabaseRestore();

  /**
   * @brief Retrieves the resource path for the database restore icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseRestoreIconPath();

  /**
   * @brief Retrieves the icon for loading data.
   * @return The 'Load Data' QIcon.
   */
  static QIcon LoadData();

  /**
   * @brief Retrieves the resource path for the load data icon.
   * @return A QString containing the file path.
   */
  static QString LoadDataIconPath();

  /**
   * @brief Retrieves the icon for database comparison.
   * @return The 'Comparison' QIcon.
   */
  static QIcon DatabaseComparison();

  /**
   * @brief Retrieves the resource path for the database comparison icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseComparisonIconPath();

  /**
   * @brief Retrieves the icon for database migration.
   * @return The 'Migration' QIcon.
   */
  static QIcon DatabaseMigration();

  /**
   * @brief Retrieves the resource path for the database migration icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseMigrationIconPath();

  /**
   * @brief Retrieves the icon for database monitoring.
   * @return The 'Monitoring' QIcon.
   */
  static QIcon DatabaseMonitorig();

  /**
   * @brief Retrieves the resource path for the database monitoring icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseMonitorigIconPath();

  /**
   * @brief Retrieves the icon for SELECT statements.
   * @return The 'SELECT' QIcon.
   */
  static QIcon DatabaseSELECT();

  /**
   * @brief Retrieves the resource path for the SELECT statement icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseSELECTIconPath();

  /**
   * @brief Retrieves the icon for UPDATE statements.
   * @return The 'UPDATE' QIcon.
   */
  static QIcon DatabaseUPDATE();

  /**
   * @brief Retrieves the resource path for the UPDATE statement icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseUPDATEIconPath();

  /**
   * @brief Retrieves the icon for INSERT statements.
   * @return The 'INSERT' QIcon.
   */
  static QIcon DatabaseINSERT();

  /**
   * @brief Retrieves the resource path for the INSERT statement icon.
   * @return A QString containing the file path.
   */
  static QString DatabaseINSERTIconPath();

  /**
   * @brief Retrieves the icon for SQL files.
   * @return The 'SQL File' QIcon.
   */
  static QIcon FileSQL();

  /**
   * @brief Retrieves the resource path for the SQL file icon.
   * @return A QString containing the file path.
   */
  static QString FileSQLIconPath();

  /**
   * @brief Retrieves the icon for text files.
   * @return The 'TXT File' QIcon.
   */
  static QIcon FileTXT();

  /**
   * @brief Retrieves the resource path for the text file icon.
   * @return A QString containing the file path.
   */
  static QString FileTXTIconPath();

  /**
   * @brief Retrieves the icon for CSS files.
   * @return The 'CSS File' QIcon.
   */
  static QIcon FileCSS();

  /**
   * @brief Retrieves the resource path for the CSS file icon.
   * @return A QString containing the file path.
   */
  static QString FileCSSIconPath();

  /**
   * @brief Retrieves the icon for HTML files.
   * @return The 'HTML File' QIcon.
   */
  static QIcon FileHTML();

  /**
   * @brief Retrieves the resource path for the HTML file icon.
   * @return A QString containing the file path.
   */
  static QString FileHTMLIconPath();

  /**
   * @brief Retrieves the icon for word wrap.
   * @return The 'Word Wrap' QIcon.
   */
  static QIcon WordWrap();

  /**
   * @brief Retrieves the resource path for the word wrap icon.
   * @return A QString containing the file path.
   */
  static QString WordWrapIconPath();

  /**
   * @brief Retrieves a string containing the first startup tip.
   * @return A QString with the tip text.
   */
  static QString StartupTip1();

  /**
   * @brief Retrieves the icon for PEM private keys.
   * @return The 'PEM Key' QIcon.
   */
  static QIcon PEMKey();

  /**
   * @brief Retrieves the resource path for the PEM key icon.
   * @return A QString containing the file path.
   */
  static QString PEMKeyIconPath();

  /**
   * @brief Retrieves the icon for PEM certificates.
   * @return The 'PEM Cert' QIcon.
   */
  static QIcon PEMCert();

  /**
   * @brief Retrieves the resource path for the PEM certificate icon.
   * @return A QString containing the file path.
   */
  static QString PEMCertIconPath();

  /**
   * @brief Retrieves the generic key icon.
   * @return The 'Key' QIcon.
   */
  static QIcon Key();

  /**
   * @brief Retrieves the resource path for the key icon.
   * @return A QString containing the file path.
   */
  static QString KeyIconPath();

  /**
   * @brief Retrieves the icon for navigation up.
   * @return The 'Go Up' QIcon.
   */
  static QIcon GoUp();

  /**
   * @brief Retrieves the resource path for the up navigation icon.
   * @return A QString containing the file path.
   */
  static QString GoUpIconPath();

  /**
   * @brief Retrieves the icon for navigation down.
   * @return The 'Go Down' QIcon.
   */
  static QIcon GoDown();

  /**
   * @brief Retrieves the resource path for the down navigation icon.
   * @return A QString containing the file path.
   */
  static QString GoDownIconPath();

  /**
   * @brief Retrieves the icon for navigation next.
   * @return The 'Go Next' QIcon.
   */
  static QIcon GoNext();

  /**
   * @brief Retrieves the resource path for the next navigation icon.
   * @return A QString containing the file path.
   */
  static QString GoNextIconPath();

  /**
   * @brief Retrieves the icon for navigation previous.
   * @return The 'Go Previous' QIcon.
   */
  static QIcon GoPrevious();

  /**
   * @brief Retrieves the resource path for the previous navigation icon.
   * @return A QString containing the file path.
   */
  static QString GoPreviousIconPath();

  /**
   * @brief Retrieves the icon for jumping/skipping navigation.
   * @return The 'Jump' QIcon.
   */
  static QIcon GoJump();

  /**
   * @brief Retrieves the resource path for the jump navigation icon.
   * @return A QString containing the file path.
   */
  static QString GoJumpIconPath();
};

#endif // DICON_H
