/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DHELPVIEWER_H
#define DHELPVIEWER_H

#include <QTextBrowser>
#include <QHelpEngine>

#include "dmdisubwindow.h"

class DTitleLabel;
class QTabWidget;
class QTextBrowser;

/**
 * @brief A customized text browser for displaying help documentation.
 *
 * This class extends QTextBrowser to correctly resolve and load resources
 * (such as images and stylesheets) directly from the compressed QHelpEngine data.
 */
class HelpBrowser : public QTextBrowser
{
public:
  /**
   * @brief Constructs the HelpBrowser.
   * @param helpEngine Pointer to the Qt Help Engine containing the documentation data.
   * @param parent The parent widget.
   */
  HelpBrowser(QHelpEngine* helpEngine, QWidget* parent = 0);

  /**
   * @brief Loads a resource (e.g., an image) from the help engine.
   *
   * Overrides the standard QTextBrowser loadResource method to fetch data
   * from the QHelpEngine instead of the file system.
   *
   * @param type The type of resource to load (e.g., Image, StyleSheet).
   * @param name The URL of the resource.
   * @return A QVariant containing the resource data.
   */
  QVariant loadResource(int type, const QUrl& name);

private:
  QHelpEngine *helpEngine;
};

/**
 * @brief The main window class for viewing application documentation.
 *
 * This class inherits from DMdiSubWindow and integrates the Qt Help Framework.
 * It provides a tabbed interface for content, indexing, and search capabilities.
 */
class DHelpViewer : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the DHelpViewer window.
   * @param collectionFileName The path to the help collection file (.qhc).
   * @param documentationFileName The path to the compressed documentation file (.qch).
   */
  DHelpViewer(QString collectionFileName, QString documentationFileName);

  /**
   * @brief Retranslates the user interface.
   *
   * Updates titles and labels to reflect the current application language.
   */
  void retranslateUI();

public slots:
  /**
   * @brief Handles search requests triggered by the search engine.
   *
   * Processes the search results and updates the view to display relevant topics.
   */
  void helpSearchEngineSearchSlot();

private:
  DTitleLabel *dTitleLabel;
  QTabWidget *tabWidget;
  HelpBrowser *helpBrowser;
  QHelpSearchEngine *helpSearchEngine;
};

#endif // DHELPVIEWER_H
