/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DFILESELECTOR_H
#define DFILESELECTOR_H

#include <QWidget>
#include <QSettings>
#include <QIcon>

#include "dicon.h"
#include "dsettings.h"

class DLineEdit;
class QDir;
class QLabel;

/**
 * @brief Helper class defining context types for file selection.
 */
class DFileSelectorContexts
{
public:
  /**
   * @brief Enumeration representing the specific purpose of the file selector.
   *
   * This determines the file filters (e.g., .sql, .png) and default paths
   * used when opening the file dialog.
   */
  enum FileSelectorContext {BackupFile = 1, LicenseFile, ProjectFile, Image, PEMKey, PEMCert, Directory, LoadDataFile};
};

/**
 * @brief A composite widget for file or directory selection.
 *
 * This widget combines a line edit (displaying the path) and a button/mechanism
 * to trigger a standard QFileDialog. It manages persistent settings for
 * remembering the last used directory based on the context.
 */
class DFileSelector : public QWidget
{
  Q_OBJECT

signals:
  /**
   * @brief Emitted when the selected file path changes.
   */
  void changed();

public:
  /**
   * @brief Constructs the DFileSelector widget.
   * @param context The type of file being selected (affects filters and settings).
   * @param label Optional text label displayed next to the selector.
   * @param showSaveDialog If true, opens a "Save As" dialog; otherwise, opens an "Open" dialog.
   * @param icon The icon to display on the selection trigger.
   */
  DFileSelector(DFileSelectorContexts::FileSelectorContext context, QString label = QString(), bool showSaveDialog = false, QIcon icon = DIcon::Folder());

  /**
   * @brief Checks if the file path is empty.
   * @return True if no file is selected.
   */
  bool isEmpty();

  /**
   * @brief Retrieves the currently selected file path.
   * @return A string containing the absolute path.
   */
  QString getFileName();

  /**
   * @brief Retrieves the directory associated with the selected file.
   * @return A QDir object pointing to the file's parent directory.
   */
  QDir fileDir();

  /**
   * @brief Sets the file name programmatically.
   * @param getFileName The full path to set.
   */
  void setFileName(QString getFileName = QString());

  /**
   * @brief Retrieves the raw text visible in the input field.
   * @return The text string.
   */
  QString fileText();

  /**
   * @brief Sets the text in the input field.
   * @param name The text to display.
   */
  void setText(QString name = QString());

  /**
   * @brief Retranslates the user interface.
   *
   * Updates dynamic text elements (like dialog titles) based on current language.
   */
  void retranslateUI();

  /**
   * @brief Sets the placeholder text for the input field.
   * @param text The placeholder string (e.g., "Select a file...").
   */
  void setPlaceholderText(QString text = QString());

  /**
   * @brief Retrieves the text from the input field.
   * @return The current text.
   */
  QString getText();

  /**
   * @brief Retrieves the current placeholder text.
   * @return The placeholder string.
   */
  QString placeholderText();

private:
  DLineEdit *lineEditFile;
  DSettings settings;
  QString setting;
  QString title;
  bool showSaveDialog;
  QLabel *fileLabel;
  DFileSelectorContexts::FileSelectorContext context;

private slots:
  /**
   * @brief Opens the appropriate QFileDialog based on the context.
   */
  void selectFileSlot();

  /**
   * @brief Internal slot to forward the changed signal.
   */
  void emitChanged();
};

#endif // DFILESELECTOR_H
