/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QFileDialog>
#include <QBoxLayout>
#include <QPushButton>
#include <QLabel>
#include <QDir>

#include "dfileselector.h"
#include "dlineedit.h"

DFileSelector::DFileSelector(DFileSelectorContexts::FileSelectorContext context, QString label, bool showSaveDialog, QIcon icon)
{
  this->showSaveDialog = showSaveDialog;
  this->context = context;
  switch(context) {
  case DFileSelectorContexts::BackupFile:
    setting = "Backup/LastFile";
    break;
  case DFileSelectorContexts::LicenseFile:
    setting = "License/File";
    break;
  case DFileSelectorContexts::ProjectFile:
    setting = "Project/File";
    break;
  case DFileSelectorContexts::Image:
    setting = "LastFileImg";
    break;
  case DFileSelectorContexts::PEMKey:
    setting = "LastPEMKeyFile";
    break;
  case DFileSelectorContexts::PEMCert:
    setting = "LastPEMCertFile";
    break;
  case DFileSelectorContexts::Directory:
    setting = "LastDirectory";
    break;
  case DFileSelectorContexts::LoadDataFile:
    setting = "LastLoadedFile";
    break;
  // default: Q_ASSERT(false);
  }
  QHBoxLayout *fileLayout = new QHBoxLayout;
  lineEditFile = new DLineEdit(icon);
  connect(lineEditFile, SIGNAL(textChanged(QString)), this, SLOT(emitChanged()));
  connect(lineEditFile, SIGNAL(clicked()), this, SLOT(selectFileSlot()));
  fileLabel = new QLabel;
  setText(label);
  fileLabel->setBuddy(lineEditFile);
  fileLayout->addWidget(fileLabel);
  fileLayout->addWidget(lineEditFile);
  fileLayout->setContentsMargins(0, 0, 0, 0);
  setLayout(fileLayout);
  retranslateUI();
}

void DFileSelector::retranslateUI()
{
  switch(context) {
  case DFileSelectorContexts::Directory:
    title = tr("Directory");
    break;
  case DFileSelectorContexts::BackupFile:
    title = tr("SQL files (*.sql)");
    break;
  case DFileSelectorContexts::LicenseFile:
    title = tr("Text files (*.txt)");
    break;
  case DFileSelectorContexts::ProjectFile:
    title = tr("Project files (*.xml)");
    break;
  case DFileSelectorContexts::Image:
    title = tr("Image files (*.png *.jpg *.bmp)");
    break;
  case DFileSelectorContexts::PEMKey:
  case DFileSelectorContexts::PEMCert:
    title = tr("PEM files (*.pem)");
    break;
  case DFileSelectorContexts::LoadDataFile:
    title = tr("All files (*.*)");
    break;
  // default: Q_ASSERT(false);
  }
}

void DFileSelector::setPlaceholderText(QString text)
{
  lineEditFile->setPlaceholderText(text);
}

QString DFileSelector::getText()
{
  return fileLabel->text();
}

QString DFileSelector::placeholderText()
{
  return lineEditFile->placeholderText();
}

bool DFileSelector::isEmpty()
{
  return lineEditFile->text().isEmpty();
}

QString DFileSelector::getFileName()
{
  return lineEditFile->text().trimmed();
}

QDir DFileSelector::fileDir()
{
  QFileInfo file(lineEditFile->text());
  return file.dir();
}

void DFileSelector::emitChanged()
{
  emit changed();
}

void DFileSelector::selectFileSlot()
{
  if (context == DFileSelectorContexts::Directory) {
    lineEditFile->setText(QFileDialog::getExistingDirectory(this, tr("Open directory"), "", QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks));
  } else {
  if (showSaveDialog)
    lineEditFile->setText(QFileDialog::getSaveFileName(this, tr("Open a file"), "", title));
  else
    lineEditFile->setText(QFileDialog::getOpenFileName(this, tr("Open a file"), "", title));
  }
  if (settings.value("GeneralSettings/OpenLastFile", false).toBool())
    settings.setValue(setting, lineEditFile->text());
  lineEditFile->setToolTip(lineEditFile->text());
}

void DFileSelector::setFileName(QString text)
{
  lineEditFile->setText(text);
  lineEditFile->setToolTip(lineEditFile->text());
}

QString DFileSelector::fileText()
{
  QFile fileToOpen(lineEditFile->text());
  bool opened = fileToOpen.open(QFile::ReadOnly | QFile::Text);
  Q_UNUSED(opened);
//  if (!fileToOpen.open(QFile::ReadOnly | QFile::Text))
//    parentWidget->statusBarMessage(tr("Cannot read file %1:\n%2.").arg(fileName).arg(fileToOpen.errorString()), QSystemTrayIcon::Critical);
  QString output(QString::fromUtf8(fileToOpen.readAll()));
  fileToOpen.close();
  return output;
}

void DFileSelector::setText(QString name)
{
  fileLabel->setText(name);
  fileLabel->setToolTip(fileLabel->text());
}
