/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DDASHBOARD_H
#define DDASHBOARD_H

#include "dbms.h"
#include "dmdisubwindow.h"

class QTimer;
//class QQuickWidget;
class DTitleLabel;
class QPieSeries;
class QChart;
class DChartView;
class QPieSlice;
class QBrush;
class QSplineSeries;
class QValueAxis;
class QPushButton;
class QGridLayout;

/**
 * @brief The DDashboard class provides a graphical monitoring interface.
 *
 * This class inherits from DMdiSubWindow and visualizes real-time database
 * server statistics. It utilizes charts (Pie, Spline) to display data such
 * as network traffic, query volume, memory usage, and storage distribution,
 * updated at various time intervals.
 */
class DDashboard : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the DDashboard window.
   * @param serverConnection Pointer to the active DBMS connection used to fetch statistics.
   */
  DDashboard(DBMS *serverConnection);

  /**
   * @brief Retranslates the user interface.
   *
   * Updates chart titles and labels to reflect the current language settings.
   */
  void retranslateUI();

  /**
   * @brief Destructor.
   *
   * Cleans up resources and stops monitoring timers.
   */
  ~DDashboard();

private:
  QTimer *qTimer100Seconds;
  QTimer *qTimer10Seconds;
  QTimer *qTimer1Second;
//  QQuickWidget *widMain;
  DBMS *serverConnection;
  QList<QStringList> *result;
  DTitleLabel *dTitleLabel;
  QPieSeries *userPieSeries;
  QChart *userChart;
  DChartView *userChartView;
  QPieSeries *databasesSizesPieSeries;
  QChart *databasesSizesChart;
  DChartView *databasesSizesChartView;
  QBrush originalSliceBrush;
  QSplineSeries *bytesSentSplineSeries;
  QSplineSeries *bytesReceivedSplineSeries;
  QChart *bytesSentOrReceivedChart;
  DChartView *bytesSentOrReceivedChartView;
  qulonglong bytesSentCurrent;
  qulonglong bytesSentLast;
  qulonglong bytesReceivedCurrent;
  qulonglong bytesReceivedLast;
  QList<QPointF> bytesSentLinePoints;
  QList<QPointF> bytesReceivedLinePoints;
  unsigned int bytesSentSplineSeriesCounter;
  QSplineSeries *queriesSplineSeries;
  QChart *queriesChart;
  DChartView *queriesChartView;
  qulonglong queriesCurrent;
  qulonglong queriesLast;
  QList<QPointF> queriesPoints;
  unsigned int queriesSplineSeriesCounter;
  int tickCount;
  ThreadedQueryExecution *threadedQueryExecution;
  bool oneThreadIsRunning;
  QPushButton *changeViewPushButton;
  QGridLayout *mainGridLayout;
  QWidget *widMain;
  QPieSeries *memoryPieSeries;
  QChart *memoryChart;
  DChartView *memoryChartView;
  QWidget *widCharts1;
  QWidget *widCharts2;
  QPieSeries *loadAveragePieSeries;
  QChart *loadAverageChart;
  DChartView *loadAverageChartView;

private slots:
  /**
   * @brief Refreshes data that changes infrequently (every 100 seconds).
   *
   * Typically used for heavy calculations like database storage sizes.
   */
  void reloadData100Seconds();

  /**
   * @brief Refreshes data that changes moderately (every 10 seconds).
   *
   * Typically used for memory usage or active user counts.
   */
  void reloadData10Seconds();

  /**
   * @brief Refreshes real-time data (every 1 second).
   *
   * Used for network traffic (bytes sent/received) and query throughput.
   */
  void reloadData1Second();

  /**
   * @brief Handles interactions when a pie chart slice is clicked.
   * @param slice The slice that was clicked (e.g., to explode or highlight it).
   */
  void handleSliceClicked(QPieSlice *slice);

  /**
   * @brief Handles interactions when the mouse hovers over a pie chart slice.
   * @param slice The slice being hovered.
   * @param state True if the mouse entered the slice, False if it left.
   */
  void handleSliceHovered(QPieSlice *slice, bool state);

  /**
   * @brief Processes the result data returned by the background thread.
   * @param result Pointer to the list of string lists containing the query results.
   */
  void handleResultsOfThreadedQueryExecution(QList<QStringList> *result);

  /**
   * @brief Toggles the visibility of a series when its legend marker is clicked.
   */
  void userChartLegendMarkerClicked();

  /**
   * @brief Toggles the dashboard layout.
   *
   * Switches the view between different layout configurations (e.g., Grid vs. Tabs).
   */
  void changeViewPushButtonClicked();

  /**
   * @brief Slot called when the threaded query execution finishes.
   *
   * Resets the `oneThreadIsRunning` flag to allow the next scheduled query to run.
   */
  void handleFinishedOfThreadedQueryExecution();

  /**
   * @brief Handles double-click events propagated from child chart views.
   *
   * Often used to maximize a specific chart or show detailed information.
   *
   * @param objectName The name of the chart object that was double-clicked.
   */
  void mouseDoubleClickEventEmitedSlot(QString objectName);
};

#endif // DDASHBOARD_H
