/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DBMS_H
#define DBMS_H

#include <QList>
#include <QSystemTrayIcon>
#include <QDateTime>
#include <QElapsedTimer>
#include <QThread>

#if defined(Q_OS_WIN)
  #include "mysql.h"
#elif defined(Q_OS_LINUX)
  #include "mysql.h"
#elif defined(Q_OS_MAC)
  #include "mysql.h"
#endif

#include "staticfunctions.h"
#include "dsettings.h"

class QVariant;
class QString;
class DBMS;
class QErrorMessage;
class QSqlTableModel;
class QSqlQueryModel;
class DQueryLogger;

/**
 * @brief The ThreadedQueryExecution class runs SQL queries in a separate thread.
 *
 * This prevents the main GUI thread from freezing during long-running database operations.
 */
class ThreadedQueryExecution : public QThread
{
  Q_OBJECT

//  Q_PROPERTY(bool onlyExecuteSafeStatements READ getOnlyExecuteSafeStatements WRITE setOnlyExecuteSafeStatements CONSTANT)
//  Q_PROPERTY(QString outputType READ getOutputType WRITE setOutputType CONSTANT)
//  Q_PROPERTY(bool saveToFile READ getSaveToFile WRITE setSaveToFile CONSTANT)
//  Q_PROPERTY(bool showNewLines READ getShowNewLines WRITE setShowNewLines CONSTANT)

  void run() override;

public:
  /**
   * @brief Constructs a threaded query executor.
   * @param serverConnectionParameters Map of connection details (host, user, etc.).
   * @param query The SQL query string to execute.
   */
  ThreadedQueryExecution(QMap<QString, QVariant> serverConnectionParameters, QString query);

  ~ThreadedQueryExecution();

  /**
   * @brief Checks if only safe statements (non-destructive) are allowed.
   * @return True if safety mode is on.
   */
  bool getOnlyExecuteSafeStatements();

  /**
   * @brief Sets the safety mode for query execution.
   * @param onlyExecuteSafeStatements True to restrict destructive queries.
   */
  void setOnlyExecuteSafeStatements(bool onlyExecuteSafeStatements);

  /**
   * @brief Gets the current output format type.
   * @return A string representing the format (e.g., "HTML", "Table").
   */
  QString getOutputType();

  /**
   * @brief Sets the format for the query result output.
   * @param outputType The desired format string.
   */
  void setOutputType(QString outputType);

  /**
   * @brief Checks if the result should be saved to a file.
   * @return True if file saving is enabled.
   */
  bool getSaveToFile();

  /**
   * @brief Enables or disables saving results to a file.
   * @param saveToFile True to enable.
   */
  void setSaveToFile(bool saveToFile);

  /**
   * @brief Checks if newlines are shown in the output.
   * @return True if newlines are visible.
   */
  bool getShowNewLines();

  /**
   * @brief Toggles the visibility of newline characters in the output.
   * @param showNewLines True to show newlines.
   */
  void setShowNewLines(bool showNewLines);

  /**
   * @brief Gets the SQL query to be executed.
   * @return The query string.
   */
  QString getQuery();

  /**
   * @brief Sets the SQL query to be executed.
   * @param query The SQL string.
   */
  void setQuery(QString query);

private:
  // DBMS *serverConnection;
  QString query;
  bool onlyExecuteSafeStatements;
  QString outputType;
  bool saveToFile;
  bool showNewLines;
  QMap<QString, QVariant> serverConnectionParameters;

signals:
  /**
   * @brief Emitted when a text-based result is ready.
   * @param result The formatted result string.
   */
  void stringResultReady(const QString &result);

  /**
   * @brief Emitted when a list-based result is ready (rows and columns).
   * @param result Pointer to the list of string lists.
   */
  void listResultReady(QList<QStringList> *result);

  /**
   * @brief Emitted when an error occurs during execution.
   * @param error The error message.
   */
  void errorOccurred(const QString error);
};

/**
 * @brief Manages database schema operations.
 */
class Databases : public QObject
{
  Q_OBJECT

public:
  Databases(DBMS *serverConnection);

  /**
   * @brief Lists all databases available on the server.
   * @return A list of database names.
   */
  QStringList list();

  /**
   * @brief Gets the CREATE DATABASE statement.
   * @param formalDatabaseName The name of the database.
   * @return The SQL creation string.
   */
  QString getDefinition(QString formalDatabaseName);

private:
  DBMS *serverConnection;
};

/**
 * @brief Manages stored procedures operations.
 */
class Procedures : public QObject
{
  Q_OBJECT

public:
  Procedures(DBMS *serverConnection);

  /**
   * @brief Lists stored procedures.
   * @param databaseName Optional database filter.
   * @return A list of procedure names.
   */
  QStringList list(QString databaseName = QString());

  /**
   * @brief Gets the CREATE PROCEDURE statement.
   * @param formalProcedureName The name of the procedure.
   * @return The SQL definition string.
   */
  QString getDefinition(QString formalProcedureName);

private:
  DBMS *serverConnection;
};

/**
 * @brief Manages stored functions operations.
 */
class Functions : public QObject
{
  Q_OBJECT

public:
  Functions(DBMS *serverConnection);

  /**
   * @brief Lists stored functions.
   * @param databaseName Optional database filter.
   * @return A list of function names.
   */
  QStringList list(QString databaseName = QString());

  /**
   * @brief Gets the CREATE FUNCTION statement.
   * @param formalFunctionName The name of the function.
   * @return The SQL definition string.
   */
  QString getDefinition(QString formalFunctionName);

private:
  DBMS *serverConnection;
};

/**
 * @brief Manages scheduled events operations.
 */
class Events : public QObject
{
  Q_OBJECT

public:
  Events(DBMS *serverConnection);

  /**
   * @brief Lists scheduled events.
   * @param databaseName Optional database filter.
   * @return A list of event names.
   */
  QStringList list(QString databaseName = QString());

  /**
   * @brief Gets the CREATE EVENT statement.
   * @param formalEventName The name of the event.
   * @return The SQL definition string.
   */
  QString getDefinition(QString formalEventName);

private:
  DBMS *serverConnection;
};

/**
 * @brief Manages database views operations.
 */
class Views : public QObject
{
  Q_OBJECT

public:
  Views(DBMS *serverConnection);

  /**
   * @brief Lists views.
   * @param databaseName Optional database filter.
   * @return A list of view names.
   */
  QStringList list(QString databaseName = QString());

  /**
   * @brief Gets the CREATE VIEW statement.
   * @param formalViewName The name of the view.
   * @return The SQL definition string.
   */
  QString getDefinition(QString formalViewName);

private:
  DBMS *serverConnection;
};

/**
 * @brief Manages database tables operations.
 */
class Tables : public QObject
{
  Q_OBJECT

public:
  Tables(DBMS *serverConnection);

  /**
   * @brief Lists tables.
   * @param databaseName Optional database filter.
   * @return A list of table names.
   */
  QStringList list(QString databaseName = QString());

  /**
   * @brief Gets the CREATE TABLE statement.
   * @param formalTableName The name of the table.
   * @return The SQL definition string.
   */
  QString getDefinition(QString formalTableName);

  /**
   * @brief Generates INSERT statements for the table's data.
   * @param formalTableName The name of the table.
   * @param returnFormalName Whether to use formal naming in the SQL.
   * @return A list of INSERT statements.
   */
  QStringList getTableDataToInsert(QString formalTableName, bool returnFormalName = true);

private:
  DBMS *serverConnection;
};

/**
 * @brief Manages triggers operations.
 */
class Triggers : public QObject
{
  Q_OBJECT

public:
  Triggers(DBMS *serverConnection);

  /**
   * @brief Lists triggers.
   * @param databaseName Optional database filter.
   * @return A list of trigger names.
   */
  QStringList list(QString databaseName = QString());

  /**
   * @brief Gets the CREATE TRIGGER statement.
   * @param formalTriggerName The name of the trigger.
   * @return The SQL definition string.
   */
  QString getDefinition(QString formalTriggerName);

private:
  DBMS *serverConnection;
};

/**
 * @brief Handles database transactions (ACID).
 */
class Transaction : public QObject
{
  Q_OBJECT

public:
  Transaction(DBMS *serverConnection);

private:
  DBMS *serverConnection;

public slots:
  /**
   * @brief Starts a new transaction.
   */
  void beginTransacction();

  /**
   * @brief Commits the current transaction.
   */
  void commitTransacction();

  /**
   * @brief Rolls back the current transaction.
   */
  void rollbackTransacction();

};

/**
 * @brief Manages Master-Slave replication control and status.
 */
class Replication : public QObject
{
  Q_OBJECT

public:
  Replication(DBMS *serverConnection);

public slots:
  /**
   * @brief Retrieves the status of the slave replication.
   * @return Textual status of the slave.
   */
  QString getSlaveStatus();

  /**
   * @brief Changes the master connection for the current session/server.
   * @param masterConnectionName The name of the new master connection.
   */
  void changeDefaultMasterConnection(QString masterConnectionName);

  /**
   * @brief Skips a specified number of replication errors.
   * @param count The number of errors to skip.
   */
  void skipErrors(unsigned int count);

  /**
   * @brief Stops the slave replication thread.
   */
  void stopSlave();

  /**
   * @brief Starts the slave replication thread.
   */
  void startSlave();

  /**
   * @brief Reboots the slave (stop and start).
   */
  void rebootSlave();

  /**
   * @brief Resets the slave replication configuration.
   */
  void resetSlave();

  /**
   * @brief Purges binary logs.
   */
  void purgeBinaryLogs();

  /**
   * @brief Flushes the relay logs.
   */
  void flushRelayLogs();

  /**
   * @brief Stops all configured slave threads.
   */
  void stopAllSlaves();

  /**
   * @brief Starts all configured slave threads.
   */
  void startAllSlaves();

  /**
   * @brief Gets slave status for a specific connection.
   * @param connectionName Name of the specific connection.
   * @return Textual status.
   */
  QString getSlaveStatus(QString connectionName);

  /**
   * @brief Retrieves a list of configured slave names.
   * @return List of slave identifiers.
   */
  QStringList getSlavesNames();

  /**
   * @brief Stops a specific slave connection.
   * @param connectionName The slave to stop.
   */
  void stopSlave(QString connectionName);

  /**
   * @brief Starts a specific slave connection.
   * @param connectionName The slave to start.
   */
  void startSlave(QString connectionName);

  /**
   * @brief Reboots a specific slave connection.
   * @param connectionName The slave to reboot.
   */
  void rebootSlave(QString connectionName);

  /**
   * @brief Resets a specific slave connection.
   * @param connectionName The slave to reset.
   */
  void resetSlave(QString connectionName);

  /**
   * @brief Retrieves the status of the master server.
   * @return Textual master status.
   */
  QString getMasterStatus();

private:
  DBMS *serverConnection;
};

/**
 * @brief Manages server processes and threads.
 */
class Processes : public QObject
{
  Q_OBJECT

public:
  Processes(DBMS *serverConnection);

  /**
   * @brief Retrieves the current process list.
   * @param useTable If true, returns data formatted for a table view.
   * @param userName Optional filter by username.
   * @return A list of process details.
   */
  QList<QStringList>* getProcessList(bool useTable = false, QString userName = QString());

  /**
   * @brief Gets headers for the process list table.
   * @param useTable True if formatting for table view.
   * @return List of header names.
   */
  QStringList getHeaderList(bool useTable = false);

  /**
   * @brief Kills a specific connection thread.
   * @param thread The thread ID.
   */
  void killThread(long long int thread);

  /**
   * @brief Kills the query running on a specific thread.
   * @param thread The thread ID.
   */
  void killQuery(long long int thread);

public slots:
  /**
   * @brief Kills idle threads exceeding a limit.
   * @param limit Time limit in seconds.
   */
  void killIdleThreads(unsigned int limit);

  /**
   * @brief Kills busy threads exceeding a limit.
   * @param limit Time limit in seconds.
   */
  void killBusyThreads(unsigned int limit);

private:
  DBMS *serverConnection;
  QList<QStringList> *result;
};

/**
 * @brief Enumeration wrapper for database item types.
 */
class ItemTypes
{
public:
  enum ItemType {Database = 1001, Table, User, UserHost, CatalogsTab, UsersTab, Trigger, View, Procedure, Routine, Event};
};

/**
 * @brief Represents a specific database table and provides operations for it.
 */
class Table
{
public:
  Table(DBMS *serverConnection, QString tableName, QString database = QString());

  /**
   * @brief Renames the table.
   * @param newName The new name.
   * @return True on success.
   */
  bool renameTable(QString newName);

  /**
   * @brief Changes the storage engine.
   * @param newEngine The new engine name (e.g., InnoDB).
   * @return True on success.
   */
  bool changeEngine(QString newEngine);

  /**
   * @brief Updates the table comment.
   * @param newComment The new comment text.
   * @return True on success.
   */
  bool changeComment(QString newComment);

  /**
   * @brief Changes the default collation of the table.
   * @param newCollation The new collation.
   * @return True on success.
   */
  bool changeCollation(QString newCollation);

  /**
   * @brief Drops the table.
   * @return True on success.
   */
  bool drop();

  /**
   * @brief Gets the fully qualified/escaped name of the table.
   * @return String in `database`.`table` format.
   */
  QString formalName();

  /**
   * @brief Gets the CREATE TABLE definition.
   * @param returnFormalName If true, uses formal naming in SQL.
   * @return The SQL string.
   */
  QString getDefinition(bool returnFormalName = true);

  /**
   * @brief Retrieves the list of fields (columns).
   * @return List of column names.
   */
  QStringList getFields();

  /**
   * @brief Drops an index from the table.
   * @param indexName The name of the index.
   * @return True on success.
   */
  bool dropIndex(QString indexName);

  /**
   * @brief Retrieves detailed index information.
   * @return List of index properties.
   */
  QList<QStringList>* getIndexes();

  /**
   * @brief Calculates the checksum of the table.
   * @return Checksum value.
   */
  unsigned long getChecksum();

  /**
   * @brief Gets the number of rows.
   * @return Row count.
   */
  unsigned long getRowCount();

  /**
   * @brief Gets the data length (size on disk).
   * @return Size in bytes.
   */
  unsigned long getDataLength();

private:
  DBMS *serverConnection;
  QString tableName;
  QString database;
};

/**
 * @brief Represents a database user.
 */
class User
{
public:
  User(DBMS *serverConnection, QString userName = QString());

  /**
   * @brief Deletes the user.
   * @return True on success.
   */
  bool drop();

  /**
   * @brief Gets the CREATE USER statement.
   * @return SQL string.
   */
  QString getDefinition();

  /**
   * @brief Gets the grants (permissions) for the user.
   * @return List of GRANT statements.
   */
  QStringList getGrants();

  /**
   * @brief Checks if the user has a specific grant.
   * @param grant The privilege to check.
   * @param databaseObject The object to check against.
   * @return True if granted.
   */
  bool hasGrant(QString grant, QString databaseObject);

  /**
   * @brief Checks if the user has root privileges.
   * @return True if root.
   */
  bool isRoot();

  /**
   * @brief Checks if the user is root and can grant privileges.
   * @return True if root with GRANT OPTION.
   */
  bool isRootAndGrantor();

  /**
   * @brief Retrieves a list of all users.
   * @return String list of usernames.
   */
  QStringList getUserList();

  /**
   * @brief Retrieves user and host combinations.
   * @param user Optional filter.
   * @return List of 'user'@'host' strings.
   */
  QStringList getUserHostList(QString user = QString());

private:
  DBMS *serverConnection;
  QString userName;
};

/**
 * @brief Represents a database view.
 */
class View
{
public:
  View(DBMS *serverConnection, QString viewName, QString database = QString());

  /**
   * @brief Deletes the view.
   * @return True on success.
   */
  bool drop();

  /**
   * @brief Gets the formal name.
   * @return `database`.`view`.
   */
  QString formalName();

  /**
   * @brief Gets the CREATE VIEW definition.
   * @return SQL string.
   */
  QString getDefinition();

private:
  DBMS *serverConnection;
  QString viewName;
  QString database;
};

/**
 * @brief Represents a database trigger.
 */
class Trigger
{
public:
  Trigger(DBMS *serverConnection, QString triggerName, QString database = QString());

  /**
   * @brief Deletes the trigger.
   * @return True on success.
   */
  bool drop();

  /**
   * @brief Gets the formal name.
   * @return `database`.`trigger`.
   */
  QString formalName();

  /**
   * @brief Gets the CREATE TRIGGER definition.
   * @return SQL string.
   */
  QString getDefinition();

private:
  DBMS *serverConnection;
  QString triggerName;
  QString database;
};

/**
 * @brief Represents a stored routine (Procedure or Function).
 */
class Routine
{
public:
  Routine(DBMS *serverConnection, QString routineName, QString database = QString());

  /**
   * @brief Deletes the routine.
   * @return True on success.
   */
  bool drop();

  /**
   * @brief Gets the formal name.
   * @return `database`.`routine`.
   */
  QString formalName();

  /**
   * @brief Determines if it is a PROCEDURE or FUNCTION.
   * @return Type string.
   */
  QString routineType();

  /**
   * @brief Gets the creation statement.
   * @return SQL string.
   */
  QString getDefinition();

private:
  DBMS *serverConnection;
  QString routineName;
  QString database;
};

/**
 * @brief Represents a Database schema.
 */
class Database : public QObject
{
  Q_OBJECT

public:
  Database(DBMS *serverConnection, QString databaseName);

  /**
   * @brief Deletes the database.
   * @return True on success.
   */
  bool drop();

  /**
   * @brief Creates the database.
   * @param collation Optional collation setting.
   * @return True on success.
   */
  bool create(QString collation = QString());

  /**
   * @brief Gets list of tables in this database.
   * @return List of table names.
   */
  QStringList getTables();

  /**
   * @brief Gets list of views in this database.
   * @return List of view names.
   */
  QStringList getViews();

  /**
   * @brief Gets list of triggers in this database.
   * @return List of trigger names.
   */
  QStringList getTriggers();

  /**
   * @brief Gets list of routines in this database.
   * @return List of routine names.
   */
  QStringList getRoutines();

  /**
   * @brief Gets list of events in this database.
   * @return List of event names.
   */
  QStringList getEvents();

  /**
   * @brief Counts the number of tables.
   * @return Count of tables.
   */
  unsigned long tableCount();

  /**
   * @brief Retrieves general info about the database.
   * @return List of properties.
   */
  QStringList info();

  /**
   * @brief Gets tables specifically local to this DB context.
   * @return List of table names.
   */
  QStringList getLocalTables();

  /**
   * @brief Gets the CREATE DATABASE statement.
   * @return SQL string.
   */
  QString getDefinition();

private:
  DBMS *serverConnection;
  QString databaseName;
  QString formalName();
};

/**
 * @brief Represents a scheduled database event.
 */
class DatabaseEvent
{
public:
  DatabaseEvent(DBMS *serverConnection, QString eventName, QString database = QString());

  /**
   * @brief Deletes the event.
   * @return True on success.
   */
  bool drop();

  /**
   * @brief Gets the formal name.
   * @return `database`.`event`.
   */
  QString formalName();

  /**
   * @brief Gets the CREATE EVENT definition.
   * @return SQL string.
   */
  QString getDefinition();

private:
  DBMS *serverConnection;
  QString eventName;
  QString database;
};

/**
 * @brief The core DBMS class handling connection and query execution.
 *
 * This class wraps the low-level database driver (e.g., MySQL C API) and provides
 * a high-level Qt-style interface for interacting with the database server.
 */
class DBMS : public QObject
{
  Q_OBJECT

//  Q_PROPERTY(QString userName READ getUserName WRITE setUserName CONSTANT)
//  Q_PROPERTY(QString characterSet READ getCharacterSet WRITE setCharacterSet CONSTANT)
//  Q_PROPERTY(QString database READ getDatabase WRITE setDatabase CONSTANT)
//  Q_PROPERTY(QString collation READ getConnectionCollation WRITE setCollation CONSTANT)
//  Q_PROPERTY(StaticFunctions::dbmsTypes DBMSType READ getDBMSType WRITE setDBMSType CONSTANT)
//  Q_PROPERTY(bool p_useSSL READ getUseSSL WRITE setUseSSL CONSTANT)
//  Q_PROPERTY(QString p_clientKey READ getKeyFile WRITE setKeyFile CONSTANT)
//  Q_PROPERTY(QString p_clientCert READ getCertFile WRITE setCertFile CONSTANT)

public:
  /**
   * @brief Constructs a DBMS object.
   * @param enableQueryLog Whether to log executed queries internally.
   */
  DBMS(bool enableQueryLog = true);

  ~DBMS();

  bool operator==(DBMS *serverConnection);
  bool operator!=(DBMS *serverConnection);

  /**
   * @brief Gets the current connection parameters.
   * @return A map of connection keys and values.
   */
  QMap<QString, QVariant> getConnectionParameters();

  /**
   * @brief Sets the connection parameters.
   * @param parameters A map of keys and values.
   */
  void setConnectionParameters(QMap<QString, QVariant> parameters);

  /**
   * @brief Opens the query logger window/utility.
   */
  void openTheDQueryLogger();

  /**
   * @brief Checks if query logging is enabled.
   * @return True if enabled.
   */
  bool queryLogEnabled();

  // Status and Variable Methods
  QString getGlobalStatusTable();
  QString getGlobalVariablesTable();
  QString getSessionStatusTable();
  QString getSessionVariablesTable();

  /**
   * @brief Gets the connection username.
   */
  QString getUserName();

  /**
   * @brief Sets the connection username.
   * @param name The username.
   */
  void setUserName(QString name);

  /**
   * @brief Sets a custom name for this connection profile.
   * @param name The profile name.
   */
  void setConnectionName(QString name);

  /**
   * @brief Gets client library information.
   */
  QString getLibraryInformation();

  /**
   * @brief Gets the custom connection profile name.
   */
  QString getConnectionName();

  /**
   * @brief Sets the connection collation.
   * @param collation The collation name.
   */
  void setCollation(QString collation);

  QString getConnectionCollation();
  QString getServerCollation();
  QString getFullUserName();

  QString getHostName();
  void setHostName(QString name);

  QString getPassword();
  void setPassword(QString pass);

  int getPort();
  int getConnectionId();
  void setPort(unsigned int number);

  QString getDatabase();
  void setDatabase(QString name);

  QString getSocket();
  QString getReplicationStatus();
//  QString getVersionComment();

  // Filtered status/variable retrieval
  QString getGlobalStatus(QString filter = QString());
  QString getGlobalVariables(QString filter = QString());
  QString getSessionStatus(QString filter = QString());
  QString getSessionlVariables(QString filter = QString());

  /**
   * @brief Gets the server version string.
   */
  QString getVersion();
  int unsigned getMayorVersion();
  int unsigned getMinorVersion();
  int unsigned getMicroVersion();
//  int getRelease();

  /**
   * @brief Closes the database connection.
   */
  void close();

  /**
   * @brief Opens the database connection.
   * @return True if successful.
   */
  bool open();

  /**
   * @brief Checks if the connection is currently open.
   * @return True if open.
   */
  bool isOpened();

  /**
   * @brief Executes a query and returns the result as a list of string lists.
   * @param queryToExecute The SQL query.
   * @param addHeaders If true, the first row contains column names.
   * @param emitNotificaction If true, emits signals on completion/error.
   * @return Pointer to the result set.
   */
  QList<QStringList>* runQuery(QString queryToExecute, bool addHeaders = false, bool emitNotificaction = true);

  /**
   * @brief Runs a query expecting a simple result set.
   * @param queryToExecute The SQL query.
   * @return Pointer to the result set.
   */
  QList<QStringList>* runQuerySimpleResult(QString queryToExecute);

  /**
   * @brief Runs a query and returns a single column as a list.
   * @param queryToExecute The SQL query.
   * @param addHeaders Include column header.
   * @return StringList of values.
   */
  QStringList runQuerySingleColumn(QString queryToExecute, bool addHeaders = false);

  /**
   * @brief Gets available databases.
   * @param skipMetaDatabases If true, skips system DBs like information_schema.
   * @return List of database names.
   */
  QStringList getDatabases(bool skipMetaDatabases = false);

  QStringList getUserHosts(QString user);
  QStringList getUsers();

  /**
   * @brief Returns the last error message from the DB driver.
   */
  QString lastError();

  /**
   * @brief Changes the currently selected database (USE db).
   * @param database The target database.
   * @return True on success.
   */
  bool changeDatabase(QString database);

  QString getTriggeredefinition(QString triggers, QString database = QString());
  QString getEventDefinition(QString event, QString database = QString());

  QStringList getEngines();
  QStringList getCollations();
  QList<QStringList>* getCharacterSets();

  // Output formatting methods
  /**
   * @brief Formats query output as a text table.
   * @param queryToExecute The SQL query.
   * @param printExtraInfo Include execution stats.
   * @param saveToFile Save result to file.
   * @param replaceReturns Sanitize newlines.
   * @param splitQuery Handle multiple statements.
   * @param emitNotificacion Emit signals.
   * @param delimiter Query delimiter.
   * @param alternateOutput Use alternate formatting.
   * @return The formatted string.
   */
  QString outputAsTable(QString queryToExecute, bool printExtraInfo = false, bool saveToFile = false, bool replaceReturns = true, bool splitQuery = true, bool emitNotificacion = true, QString delimiter = ";", bool alternateOutput = false);

  QString outputAsV(QString queryToExecute, bool printRowsInSet = false, bool saveToFile = false, bool replaceReturns = true, bool splitQuery = true, bool removeHeaders = false, QString delimiter = ";");
  QString outputAsVV(QString queryToExecute, bool saveToFile = false, bool replaceReturns = true, bool splitQuery = true, QString delimiter = ";");
  QString outputAsHTML(QString queryToExecute, bool saveToFile = false, bool replaceReturns = true, bool splitQuery = true, QString delimiter = ";");
  QString outputAsXML(QString queryToExecute, bool saveToFile = false, bool replaceReturns = true, bool splitQuery = true, QString delimiter = ";");
  QString outputAsG(QString queryToExecute, bool saveToFile = false, bool replaceReturns = true, bool splitQuery = true, bool emitNotificacion = true, QString delimiter = ";");
  QString outputAsJ(QString queryToExecute, bool saveToFile = false, bool replaceReturns = true, bool splitQuery = true, bool emitNotificacion = true, QString delimiter = ";");

  /**
   * @brief Tests if the connection is still alive.
   * @return True if connected.
   */
  bool testOpened();

  QStringList executedQueries;

  QString getCharacterSet();
  void setCharacterSet(QString charset = "utf8mb4");

  /**
   * @brief Shuts down the database server.
   * @return True on success.
   */
  bool shutdown();

  QString getStatus();

  /**
   * @brief Executes a query without returning a result set (e.g., UPDATE, INSERT).
   * @param queryToExecute The SQL statement.
   * @return True on success.
   */
  bool executeQuery(QString queryToExecute);

  void flushPrivileges();
  void flushHosts();

  QList<QStringList> *getPrivileges(QString filter = QString());

  /**
   * @brief Returns the SQL type string representing a string literal.
   */
  QString getStringType();

  inline QString replaceReturnsAndTabs(QString string);
  QString getConnectionString();
  QString getfailedQueries();

  void logApplicationStarted();
  void logStatement(QString statement, QString result = QString());

  // SQLite helper methods (likely for local history/settings)
  QSqlTableModel *sqliteTableModel();
  QSqlQueryModel *sqliteFilterQueryModel();
  void clearSQLiteQueryLog();

  void setCharsetAndCollation(QString charset, QString collation);
  QList<QStringList> *getCollationsApplicability();

  void setKeyFile(QString keyFile);
  void setCertFile(QString certFile);
  void setUseSSL(bool useSSL);
  bool getUseSSL();

  /**
   * @brief Pings the server.
   * @return 0 on success, non-zero on error.
   */
  int ping();

  StaticFunctions::dbmsTypes getDBMSType();
  void setDBMSType(StaticFunctions::dbmsTypes type);
  StaticFunctions::dbmsTypes p_DBMSType;
  QStringList getCharsets();

  // Factory methods for helper objects
  Table *table(QString tableName, QString database = QString());
  View *view(QString viewName, QString database = QString());
  Trigger *trigger(QString triggerName, QString database = QString());
  Routine *routine(QString routineName, QString database = QString());
  Database *database(QString databaseName = QString());
  DatabaseEvent *databaseEvent(QString eventName, QString database = QString()); // Renamed from Event to DatabaseEvent to avoid compilation warnings
  Processes *processes();
  Replication *replication();
  User *user(QString userName = QString());
  Transaction *transaction();
  Triggers *triggers();
  Tables *tables();
  Views *views();
  Events *events();
  Functions *functions();
  Procedures *procedures();
  Databases *databases();

  void saveOutputToFile(QString contents, QString filter, QString fileName = QString());
  QString lastErrorNumber();
  QString getKeyFile();
  QString getCertFile();
  void insertOnExecutedQueries(QString sessionid, QString connection, QString query, QString result);
  QString lastExecutedQuery;
  QErrorMessage *errorMessage;

  /**
   * @brief Access the raw MariaDB connection handle.
   * @return Pointer to MYSQL struct.
   */
  MYSQL *getMariadbConnection();

  bool dummyExecution(QString statement);

  /**
   * @brief Escapes a string for use in SQL queries.
   * @param input The raw string.
   * @return The escaped string.
   */
  QString escapeString(const QString &input);

signals:
  void errorOccurred();
  void databaseChanged();
  void statusBarMessage(const QString &message, QSystemTrayIcon::MessageIcon icon = QSystemTrayIcon::Information, int timeout = 0);
  void errorMessageAccepted();
  void reconnectionPerformed();
  void changeDelimiter(const QString &delimiter);

private slots:
  void errorMessageAcceptedSlot();
  void checkIfReconnected();

private:
  MYSQL *mariadbConnection;
  MYSQL_RES *mariadbResults;
  MYSQL_FIELD *field;
  MYSQL_ROW record;
  QString p_userName;
  QString p_connectionName;
  QString hostName;
  QString password;
  unsigned int port;
  QString p_database;
  bool opened;
  int query(QString queryToExecute);
//  QString outputAsTable(MYSQL_RES *result, QString query, bool replaceReturns);
//  QString outputAsTable(PGresult *result, QString query, bool replaceReturns);
  QString errorOnExecution(const QString message, const QString type, const QString statement);
  QString processOutput;
  DSettings settings;
  QList<QStringList>* failedQueries;
  void logExecutedQueries(QString query, QString result = QString());
  static void printQueryProgress(const MYSQL *mysql, uint stage, uint max_stage, double progress, const char *proc_info, uint proc_info_length);
  QElapsedTimer queryExecutionTime;
  QString millisecondsToTime(unsigned int milliseconds);
  QString p_collation;
  QString p_charset;
  bool p_useSSL;
  QString p_clientKey;
  QString p_clientCert;
  QTimer *timerCheckIfReconnected;
  ulong p_connectionId;
  DQueryLogger *dQueryLogger;
//  static inline QByteArray fromUnicode(QTextCodec *tc, const QString &str);
  bool enableQueryLog;

};

#endif // DBMS_H
