/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QErrorMessage>
#include <QFileDialog>
#include <QTextStream>
#include <QStringList>
#include <QMessageBox>
#include <QApplication>
#include <QSystemTrayIcon>
#include <QtSql/QSqlTableModel>
#include <QtSql/QSqlQueryModel>
#include <QTimer>
//#include <QTextCodec>
#include <QRegularExpression>
#include <QDir>

#include "dbms.h"
#include "staticfunctions.h"
#include "dquerylogger.h"

#include "QDebug"

DBMS::DBMS(bool enableQueryLog)
{
  opened = false;
  this->enableQueryLog = enableQueryLog;
  executedQueries = QStringList();
  failedQueries = new QList<QStringList>();
  failedQueries->append(QStringList() << tr("Count") << tr("Query") << tr("Error message"));
  errorMessage = new QErrorMessage;
  errorMessage->setWindowTitle(tr("Error"));
  connect(errorMessage, SIGNAL(accepted()), this, SLOT(errorMessageAcceptedSlot()));

  openTheDQueryLogger();
  queryExecutionTime.start();

  timerCheckIfReconnected = new QTimer(this);
  timerCheckIfReconnected->setInterval(600000);
  connect(timerCheckIfReconnected, SIGNAL(timeout()), this, SLOT(checkIfReconnected()));
  timerCheckIfReconnected->start();
}

bool DBMS::shutdown()
{
  if (mysql_shutdown(mariadbConnection, SHUTDOWN_DEFAULT) == 0) {
    opened = false;
    return true;
  }
  return false;
}

bool DBMS::open()
{
  //last_progress_report_length = 0;
  mysql_library_init(0, NULL, NULL);
  mariadbConnection = mysql_init(NULL);

  if (!mariadbConnection)
    QMessageBox::warning(0, tr("Library initialization"), tr("Error on library initialization."));

  // https://mariadb.com/kb/en/mysql_optionsv/

  mysql_optionsv(mariadbConnection, MYSQL_OPT_LOCAL_INFILE, (void *)NULL);

  my_bool reconnect = settings.value("DBMS/Reconnect", 1).toInt();
  mysql_optionsv(mariadbConnection, MYSQL_OPT_RECONNECT, &reconnect);

  int HANDLE_EXPIRED_PASSWORDS = 1;
  mysql_optionsv(mariadbConnection, MYSQL_OPT_CAN_HANDLE_EXPIRED_PASSWORDS, &HANDLE_EXPIRED_PASSWORDS);

  mysql_optionsv(mariadbConnection, MYSQL_PROGRESS_CALLBACK, (void*) printQueryProgress);

  //  mysql_optionsv(mysql, MYSQL_INIT_COMMAND, (void *)"CREATE TABLE ...");

  unsigned int timeout= 3;
  mysql_optionsv(mariadbConnection, MYSQL_OPT_CONNECT_TIMEOUT, (void *) &timeout);

//  mysql_optionsv(mariadbConnection, MYSQL_OPT_MAX_ALLOWED_PACKET, 0x40000000);

//  mysql_optionsv(mariadbConnection, MYSQL_OPT_NET_BUFFER_LENGTH, 0x40000000);

  mysql_optionsv(mariadbConnection, MYSQL_OPT_COMPRESS, NULL);

  if (p_useSSL && !p_clientCert.isEmpty() && !p_clientKey.isEmpty())
    mysql_ssl_set(mariadbConnection, p_clientKey.toUtf8().data(), p_clientCert.toUtf8().data(), NULL, NULL, NULL);

  opened = mysql_real_connect(mariadbConnection, hostName.toUtf8().data(), p_userName.toUtf8().data()
                              , password.toUtf8().data(), p_database.isEmpty() ? NULL : p_database.toUtf8().data(), port
                              , NULL, CLIENT_MULTI_STATEMENTS | CLIENT_INTERACTIVE | CLIENT_PROGRESS);
  if (opened) {
    setCharsetAndCollation(getCharacterSet(), getConnectionCollation());
    p_connectionId = runQuerySingleColumn("SELECT CONNECTION_ID()").at(0).toULong();
//    getLibraryInformation();
  }

  return opened;
}

void DBMS::printQueryProgress(const MYSQL *mysql, uint stage, uint max_stage, double progress, const char *proc_info, uint proc_info_length)
{
  Q_UNUSED(mysql);
  Q_UNUSED(proc_info_length);

  qDebug() << tr("Stage: %1 of %2 '%3'. %4% of stage done.\nTotal progess: %5.")
              .arg(stage).arg(max_stage).arg(proc_info).arg(progress)
              .arg(((stage -1) / (double) max_stage * 100.00 + progress / max_stage) * 2);

//  uint length= printf("Stage: %d of %d '%.*s' %6.3g%% of stage done",
//                        stage, max_stage, proc_info_length, proc_info,
//                        progress);
//    if (length < last_progress_report_length)
//      printf("%*s", last_progress_report_length - length, "");
//    putc('\r', stdout);
//    fflush(stdout);
  //    last_progress_report_length= length;
}

QString DBMS::millisecondsToTime(unsigned int milliseconds)
{
  float seconds = (float) (milliseconds / 1000);
  if (milliseconds < 1000)
    return tr("%1 milliseconds").arg(milliseconds);
  if (seconds <= 60)
    return tr("%1 seconds").arg(seconds);
  if (seconds > 60 && seconds < 3600)
    return tr("%1 minutes").arg(seconds / 60);
  if (seconds >= 3600)
    return tr("%1 hours").arg(seconds / 3600);
  return "0 seconds";
}

bool DBMS::queryLogEnabled()
{
  return settings.value("GeneralSettings/EnableQueryLog", false).toBool() && enableQueryLog;
}

//QByteArray DBMS::fromUnicode(QTextCodec *tc, const QString &str)
//{
//#ifdef QT_NO_TEXTCODEC
//    Q_UNUSED(tc);
//    return str.toLatin1();
//#else
//    return tc->fromUnicode(str);
//#endif
//}

void DBMS::close()
{
  logExecutedQueries(getfailedQueries());
  logExecutedQueries(tr("Application closed"));

  if (isOpened()) {
    opened = false;
    mysql_close(mariadbConnection);
    mysql_library_end();
  }
}

DBMS::~DBMS()
{
  close();
}

bool DBMS::operator==(DBMS *serverConnection)
{
  //A DBMS is considerated equal to DBMS is the Host, Port and Database are the same
  if (this->getHostName() == serverConnection->getHostName())
    if (this->getPort() == serverConnection->getPort())
      if (this->getDatabase() == serverConnection->getDatabase())
        return true;
  return false;
}

bool DBMS::operator!=(DBMS *serverConnection)
{
  return !operator==(serverConnection);
}

QMap<QString, QVariant> DBMS::getConnectionParameters()
{
  QMap<QString, QVariant> connectionParameters;
  connectionParameters.insert("ConnectionType", getDBMSType());
  connectionParameters.insert("User", getUserName());
  connectionParameters.insert("Host", getHostName());
  connectionParameters.insert("Port", getPort());
  connectionParameters.insert("Database", getDatabase());
//  connectionParameters.insert("ConnectionsCount", 0);
  connectionParameters.insert("Collation", getCharacterSet() + "|" + getConnectionCollation());
  connectionParameters.insert("UseSSL", getUseSSL());
  connectionParameters.insert("KeyFile", getKeyFile());
  connectionParameters.insert("CertFile", getCertFile());
  connectionParameters.insert("Password", StaticFunctions::password(getPassword(), true));
//  connectionParameters.insert("Name", connectionName);
  return connectionParameters;
}

void DBMS::setConnectionParameters(QMap<QString, QVariant> parameters)
{
  if (parameters.value("ConnectionType") == "--")
    setDBMSType(StaticFunctions::Undefined);
  if (parameters.value("ConnectionType")  == "MySQL")
    setDBMSType(StaticFunctions::MySQL);
  if (parameters.value("ConnectionType")  == "MariaDB")
    setDBMSType(StaticFunctions::MariaDB);
  setUserName(parameters.value("User").toString());
  setHostName(parameters.value("Host").toString());
  setPort(parameters.value("Port").toInt());
  setDatabase(parameters.value("Database").toString());
  setCharacterSet(parameters.value("Collation").toString().split("|").at(0));
  setCollation(parameters.value("Collation").toString().split("|").at(1));
  setUseSSL(parameters.value("UseSSL").toBool());
  setKeyFile(parameters.value("KeyFile").toString());
  setCertFile(parameters.value("CertFile").toString());
  setPassword(StaticFunctions::password(parameters.value("Password").toString()));
  setConnectionName(parameters.value("Name").toString());
}

void DBMS::openTheDQueryLogger()
{
  if (queryLogEnabled()) {
    dQueryLogger = new DQueryLogger(this);
  }
}

QString DBMS::getGlobalStatusTable()
{
  if (getVersion().left(3) == "5.7")
    return "`performance_schema`.`global_status`";
  else
    return "`information_schema`.`GLOBAL_STATUS`";
}

QString DBMS::getGlobalVariablesTable()
{
  if (getVersion().left(3) == "5.7")
    return "`performance_schema`.`global_variables`";
  else
    return "`information_schema`.`GLOBAL_VARIABLES`";
}

QString DBMS::getSessionStatusTable()
{
  if (getVersion().left(3) == "5.7")
    return "`performance_schema`.`session_status`";
  else
    return "`information_schema`.`SESSION_STATUS`";
}

QString DBMS::getSessionVariablesTable()
{
  if (getVersion().left(3) == "5.7")
    return "`performance_schema`.`session_variables`";
  else
    return "`information_schema`.`SESSION_VARIABLES`";
}

bool DBMS::testOpened()
{
  if (!isOpened()) {
    emit statusBarMessage(tr("Connection is not opened."));
    errorMessage->showMessage(tr("Connection is not opened."), "Connection");
    return false;
  }
  return true;
}

bool DBMS::isOpened()
{
  return opened;
}

void DBMS::setDatabase(QString name)
{
  p_database = name;
}

void DBMS::setHostName(QString name)
{
  hostName = name;
}

void DBMS::setPassword(QString pass)
{
  password = pass;
}

void DBMS::setPort(unsigned int number)
{
  port = number;
}

void DBMS::setUserName(QString name)
{
  p_userName = name;
}

void DBMS::setConnectionName(QString name)
{
  p_connectionName = name;
}

QString DBMS::getConnectionName()
{
  return p_connectionName;
}

QString DBMS::getLibraryInformation()
{
//  const MARIADB_CHARSET_INFO *charSetInfo;

  // https://mariadb.com/kb/en/mariadb_get_infov/

  MY_CHARSET_INFO cs;
  mysql_get_character_set_info(mariadbConnection, &cs);

  qDebug() << "mysql_get_client_info: " << mysql_get_client_info() << "\n"
           << "mysql_get_client_version: " << mysql_get_client_version() << "\n"
           << "mysql_get_proto_info: " << mysql_get_proto_info(mariadbConnection) << "\n"
           << "mysql_get_host_info: " << mysql_get_host_info(mariadbConnection) << "\n"
           << "mysql_get_server_version: " << mysql_get_server_version(mariadbConnection) << "\n"
           << "mysql_get_server_info: " << mysql_get_server_info(mariadbConnection) << "\n"

           << "character set+collation number: " << cs.number << "\n"
           << "character set name: " << cs.name << "\n"
           << "collation name: " << cs.csname << "\n"
           << "comment: " << cs.comment << "\n"
           << "directory: " << cs.dir << "\n"
           << "multi byte character min. length: " << cs.mbminlen << "\n"
           << "multi byte character max. length: " << cs.mbmaxlen << "\n"
              ;
//  << "mariadb_get_infov(MARIADB_CONNECTION_USER): " << mariadb_get_infov(mariadbConnection, MARIADB_CHARSET_NAME, (void *)&charSetInfo) << charSetInfo << "\n"
  return QString("");
}

void DBMS::setCollation(QString collation)
{
  p_collation = collation;
}

QString DBMS::getConnectionCollation()
{
  return p_collation;
}

QString DBMS::getServerCollation()
{
  if (isOpened())
    return runQuerySingleColumn("SELECT `VARIABLE_VALUE` FROM " + getGlobalVariablesTable() + " WHERE `VARIABLE_NAME` = 'COLLATION_SERVER'").at(0);
  return QString();
}

QStringList DBMS::getDatabases(bool skipMetaDatabases)
{
  if (isOpened()) {
    if (skipMetaDatabases) {
      return runQuerySingleColumn("SELECT `SCHEMA_NAME` FROM `information_schema`.`SCHEMATA` WHERE `SCHEMA_NAME` NOT IN ('information_schema', 'performance_schema', 'sys', 'mysql') ORDER BY `SCHEMA_NAME`");
    } else {
      return runQuerySingleColumn("SELECT `SCHEMA_NAME` FROM `information_schema`.`SCHEMATA` ORDER BY `SCHEMA_NAME`");
    }
  }
  return QStringList();
}

QStringList DBMS::getUsers()
{
  if (isOpened())
    return runQuerySingleColumn("SELECT CONCAT(\"'\", `User`, \"'@'\", `Host`, \"'\") FROM `mysql`.`user` ORDER BY `User`, `Host`", false);
  return QStringList();
}

QStringList DBMS::getUserHosts(QString user)
{
  if (isOpened())
    return runQuerySingleColumn(QString("SELECT `Host` FROM `mysql`.`user` WHERE `User` = '%1'").arg(user));
  return QStringList();
}

QStringList DBMS::runQuerySingleColumn(QString queryToExecute, bool addHeaders)
{
  QStringList rows;

  if (isOpened()) {
    query(queryToExecute);
    mariadbResults = mysql_store_result(mariadbConnection);
    if (!mariadbResults)
      return QStringList();
    if (mysql_field_count(mariadbConnection) == 0)
      errorOnExecution(lastError(), lastErrorNumber(), queryToExecute);
    if (addHeaders)
      while((field = mysql_fetch_field(mariadbResults)))
        rows.append(field->name);
    while((record = mysql_fetch_row(mariadbResults)))
      rows.append(record[0]);
    mysql_free_result(mariadbResults);
  }
  return rows;
}

QList<QStringList>* DBMS::runQuery(QString queryToExecute, bool addHeaders, bool emitNotificaction)
{
  queryExecutionTime.restart();
  QList<QStringList> *rows = new QList<QStringList>();
  QStringList row;
  int status = 0;
  //  qDebug() << queryToExecute;

  if (isOpened()) {
    if (queryToExecute.trimmed().startsWith("DELIMITER", Qt::CaseInsensitive)) {
      if (addHeaders) {
        row.append("DELIMITER");
        rows->append(row);
      }
      row.clear();
      static QRegularExpression re("( |\\t|\\r|\\n)");
      if (queryToExecute.trimmed().split(re, Qt::SkipEmptyParts).count() > 1)
        emit changeDelimiter(queryToExecute);
    }
    if (query(queryToExecute) == 0) {
      do {
        mariadbResults = mysql_store_result(mariadbConnection);
        if (mariadbResults) {
          if (addHeaders) {
            while((field = mysql_fetch_field(mariadbResults)))
              row.append(field->name);
            rows->append(row);
            row.clear();
          }
          while((record = mysql_fetch_row(mariadbResults))) {
            for (int counter = 0; counter < (int) mysql_num_fields(mariadbResults); counter++)
              row.append(QString::fromUtf8(record[counter]));
            rows->append(row);
            row.clear();
          }
          rows->append(QStringList() << tr("Rows in set: %1. Elapsed time: %2.").arg(mysql_affected_rows(mariadbConnection)).arg(millisecondsToTime(queryExecutionTime.elapsed())));
          mysql_free_result(mariadbResults);
        } else {/* no result set or error */
          if (mysql_field_count(mariadbConnection) == 0) {
            //rows->append(QStringList() <<tr("Rows in set: %1").arg(mysql_affected_rows(mariadbConnection)));
            rows->append(QStringList() << tr("Rows in set: %1. Elapsed time: %2.").arg(mysql_affected_rows(mariadbConnection)).arg(millisecondsToTime(queryExecutionTime.elapsed())));
          } else { /* some error occurred */
            rows->append(QStringList() << errorOnExecution(lastError(), lastErrorNumber(), queryToExecute));
          }
        }
        /* more results? -1 = no, >0 = error, 0 = yes (keep looping) */
        if ((status = mysql_next_result(mariadbConnection)) > 0) {
          rows->append(QStringList() << errorOnExecution(lastError(), lastErrorNumber(), queryToExecute));
          //rows->append(QStringList() << tr("Rows in set: %1").arg(0)); //Implement this for PSQL
          rows->append(QStringList() << tr("Rows in set: %1. Elapsed time: %2 seconds.").arg(0).arg(0)); //Implement this for PSQL
        }
      } while (status == 0);
    } else {
      rows->append(QStringList() << errorOnExecution(lastError(), lastErrorNumber(), queryToExecute));
      //rows->append(QStringList() << tr("Rows in set: %1").arg(0));
      rows->append(QStringList() << tr("Rows in set: %1. Elapsed time: %2 seconds.").arg(0).arg(0));
    }
    //  for (int counter = 0; counter < rows->count(); counter++)
    //    qDebug() << "*" << rows->at(counter);
  }
  if (emitNotificaction)
    emit statusBarMessage(rows->last().at(0));
  return rows;
}

QList<QStringList>* DBMS::runQuerySimpleResult(QString queryToExecute)
{
  QList<QStringList> *rows = new QList<QStringList>();
  QStringList row;
  int status = 0;

  //  qDebug() << queryToExecute;

  if (isOpened()) {
    if (query(queryToExecute) == 0) {
      do {
        /* did current statement return data? */
        mariadbResults = mysql_store_result(mariadbConnection);
        if (mariadbResults) {
          /* yes; process rows and free the result set */
          //          if (addHeaders) {
          //            while((field = mysql_fetch_field(mariadbResults)))
          //              row.append(field->name);
          //            rows->append(row);
          //            row.clear();
          //          }
          while((record = mysql_fetch_row(mariadbResults))) {
            for (int counter = 0; counter < (int) mysql_num_fields(mariadbResults); counter++)
              row.append(QString::fromUtf8(record[counter]));
            rows->append(row);
            row.clear();
          }
          //          rows->append(QStringList() << tr("Rows in set") << QString("%1").arg(mysql_affected_rows(mariadbConnection)));
          mysql_free_result(mariadbResults);
        } else {/* no result set or error */
          if (mysql_field_count(mariadbConnection) != 0) {
            //            rows->append(QStringList() << tr("Rows in set") << QString("%1").arg(mysql_affected_rows(mariadbConnection)));
            //          } else { /* some error occurred */
            rows->append(QStringList() << errorOnExecution(lastError(), lastErrorNumber(), queryToExecute));
          }
        }
        /* more results? -1 = no, >0 = error, 0 = yes (keep looping) */
        if ((status = mysql_next_result(mariadbConnection)) > 0) {
          rows->append(QStringList() << errorOnExecution(lastError(), lastErrorNumber(), queryToExecute));
          //          rows->append(QStringList() << tr("Rows in set") << "0"); //Implement this for PSQL
        }
      } while (status == 0);
    } else {
      rows->append(QStringList() << errorOnExecution(lastError(), lastErrorNumber(), queryToExecute));
      //      rows->append(QStringList() << tr("Rows in set") << "0");
    }
    //  for (int counter = 0; counter < rows->count(); counter++)
    //    qDebug() << "*" << rows->at(counter);
  }
  return rows;
}

QString DBMS::lastError()
{
  return QString("%1 %2 %3").arg(tr("Error:")).arg(mysql_errno(mariadbConnection)).arg(mysql_error(mariadbConnection));
}

int DBMS::query(QString queryToExecute)
{
  int valueToReturn = 0;
  std::string statement = queryToExecute.toStdString();
  if (isOpened()) {
    //      valueToReturn = mysql_real_query(mariadbConnection, encQuery.data(), encQuery.length());
    lastExecutedQuery = queryToExecute;
    valueToReturn = mysql_real_query(mariadbConnection, statement.c_str(), statement.size());
    if (valueToReturn != 0)
      failedQueries->insert(1, QStringList() << QString("%1").arg(failedQueries->count()) << queryToExecute << lastError());
  }
  return valueToReturn;
}

QString DBMS::outputAsTable(QString queryToExecute, bool printExtraInfo, bool saveToFile, bool replaceReturns, bool splitQuery, bool emitNotificacion, QString delimiter, bool alternateOutput)
{
  int row = 0;
  int row2 = 0;
  QList<int> maxWidthList;
  QString output;
  QStringList queryList;
  if (splitQuery)
    //queryList = queryToExecute.split(QRegularExpression("\\" + delimiter + "\\s+"), Qt::SkipEmptyParts);
    queryList = queryToExecute.split(delimiter, Qt::SkipEmptyParts);
  else
    queryList.append(queryToExecute);

  if (isOpened()) {
    foreach (QString statement, queryList) {
      QList<QStringList> *rows = runQuery(statement, true, emitNotificacion);
      //    for (int i = 0; i <= rows->count(); i++)
      //        qDebug() << rows->at(i);

      //Find the maximum values for the width of the columns
      for (row = 0; row < rows->at(0).count(); row++)
        maxWidthList.append(1);
      for (row = 0; row < (rows->count() - 1); row++)
        for (row2 = 0; row2 < rows->at(row).count(); row2++)
          if (QString(rows->at(row).at(row2)).length() > maxWidthList.at(row2))
            maxWidthList.replace(row2, QString(rows->at(row).at(row2)).length());

      if (printExtraInfo)
        output += "--------------\n" + statement + "\n--------------\n";

      //Prints the headres and theirs borders
      if (!alternateOutput) {
        output += "+";
        for (row = 0; row < rows->at(0).count(); row++)
          output +=  QString('-').repeated(maxWidthList.at(row) + 2) + "+";
        output += "\n|";
      } else {
        output += "\n";
      }
      for (row = 0; row < rows->at(0).count(); row++)
        output +=  " " + replaceReturnsAndTabs(rows->at(0).at(row)).leftJustified(maxWidthList.at(row), ' ') + ((((row + 1) == rows->at(0).count()) && alternateOutput) ? "" : " |");
      output += "\n";
      if (!alternateOutput)
        output += "+";
      for (row = 0; row < rows->at(0).count(); row++)
        output +=  QString('-').repeated(maxWidthList.at(row) + 2) + "+";
      if (alternateOutput)
        output = output.left(output.size() - 2);

      //Print the data
      if (replaceReturns) {
        for (row = 1; row < rows->count() - 1; row++) {
          output += "\n";
          if (!alternateOutput)
            output += "|";
          for (row2 = 0; row2 < rows->at(row).count(); row2++)
            output +=  " " + replaceReturnsAndTabs(rows->at(row).at(row2)).leftJustified(maxWidthList.at(row2), ' ') + ((((row2 + 1) == rows->at(0).count()) && alternateOutput) ? "" : " |");
        }
      } else {
        for (row = 1; row < rows->count() - 1; row++) {
          output += "\n";
          if (!alternateOutput)
            output += "|";
          for (row2 = 0; row2 < rows->at(row).count(); row2++)
            output +=  " " + QString::fromUtf8(rows->at(row).at(row2).toUtf8()).leftJustified(maxWidthList.at(row2), ' ') + ((((row2 + 1) == rows->at(0).count()) && alternateOutput) ? "" : " |");
        }
      }

      output += "\n";
      if (!alternateOutput) {
        //Print the botton border
        output += "+";
        for (row = 0; row < rows->at(0).count(); row++)
          output +=  QString('-').repeated(maxWidthList.at(row) + 2) + "+";
      }

      output += "\n" + rows->last().at(0) + "\n";
    }
  }

  if (saveToFile) {
    saveOutputToFile(output, "Text Files (*.txt)", settings.value("GeneralSettings/LastTextFile", "").toString());
    return "";
  }
  return output;
}

void DBMS::saveOutputToFile(QString contents, QString filter, QString fileName)
{
  fileName = QFileDialog::getSaveFileName(0, tr("Select a file"), fileName, filter);
  if (fileName.startsWith("HTML"))
    settings.value("GeneralSettings/LastHTMLFile", fileName);
  if (fileName.startsWith("XML"))
    settings.value("GeneralSettings/LastXMLFile", fileName);
  if (fileName.startsWith("Text"))
    settings.value("GeneralSettings/LastTextLFile", fileName);
  QFile file(fileName);
  if (!file.open(QFile::WriteOnly | QFile::Text))
    emit statusBarMessage(tr("Cannot write file %1:\n%2.").arg(fileName, file.errorString()));
  QTextStream out(&file);
  out << contents.toUtf8();
  file.close();
  emit statusBarMessage(tr("Data exported to %1").arg(fileName));
}

QString DBMS::outputAsHTML(QString queryToExecute, bool saveToFile, bool replaceReturns, bool splitQuery, QString delimiter)
{
  QString output("<HTML>\n<BODY>\n");
  int row = 0;
  int row2 = 0;
  QStringList queryList;
  if (splitQuery)
    queryList = queryToExecute.split(QRegularExpression("\\" + delimiter + "\\s+"), Qt::SkipEmptyParts);
  else
    queryList.append(queryToExecute);

  if (isOpened()) {
    foreach (QString statement, queryList) {
      QList<QStringList> *rows = runQuery(statement, true);

      output += "  <TABLE border='1'>\n";
      //Prints the headres and theirs borders
      output += "    <TR>\n";
      for (row = 0; row < rows->at(0).count(); row++)
        output += "      <TH>" + replaceReturnsAndTabs(rows->at(0).at(row)) + "</TH>\n";
      output += "    </TR>\n";

      //Print the data
      if (replaceReturns) {
        for (row = 1; row < (rows->count() - 1); row++) {
          output += "    <TR>\n";
          for (row2 = 0; row2 < rows->at(row).count(); row2++)
            output +=  "      <TD>" + (QString(rows->at(row).at(row2)).length() == 0 ? "&nbsp;" : replaceReturnsAndTabs(rows->at(row).at(row2)))
                + "</TD>\n";
          output += "    </TR>\n";
        }
      } else {
        for (row = 1; row < (rows->count() - 1); row++) {
          output += "    <TR>\n";
          for (row2 = 0; row2 < rows->at(row).count(); row2++)
            output +=  "      <TD>" + (QString(rows->at(row).at(row2)).length() == 0 ? "&nbsp;" : rows->at(row).at(row2))
                + "</TD>\n";
          output += "    </TR>\n";
        }
      }
      output += "  </TABLE>\n  <P>" + rows->last().at(0) + "</P>\n";
    }
  }
  output += "</BODY>\n</HTML>\n";
  if (saveToFile) {
    saveOutputToFile(output, "HTML Files (*.html)", settings.value("GeneralSettings/LastHTMLFile", "").toString());
    return "";
  } else {
    return output;
  }
}

QString DBMS::outputAsXML(QString queryToExecute, bool saveToFile, bool replaceReturns, bool splitQuery, QString delimiter)
{
  QString output;
  int row = 0;
  int row2 = 0;
  QStringList queryList;
  if (splitQuery)
    queryList = queryToExecute.split(QRegularExpression("\\" + delimiter + "\\s+"), Qt::SkipEmptyParts);
  else
    queryList.append(queryToExecute);

  if (isOpened()) {
    foreach (QString statement, queryList) {
      QList<QStringList> *rows = runQuery(statement, true);

      output += "<resultset>\n";
      //Print the data
      if (replaceReturns) {
        for (row = 1; row < rows->count() - 1; row++) {
          output += "  <row>\n";
          for (row2 = 0; row2 < rows->at(row).count(); row2++)
            output +=  "    <" + replaceReturnsAndTabs(rows->at(0).at(row2)) + ">" + replaceReturnsAndTabs(rows->at(row).at(row2)) + "</" + replaceReturnsAndTabs(rows->at(0).at(row2)) + ">\n";
          output += "  </row>\n";
        }
      } else {
        for (row = 1; row < rows->count() - 1; row++) {
          output += "  <row>\n";
          for (row2 = 0; row2 < rows->at(row).count(); row2++)
            output +=  "    <" + replaceReturnsAndTabs(rows->at(0).at(row2)) + ">" + rows->at(row).at(row2) + "</" + replaceReturnsAndTabs(rows->at(0).at(row2)) + ">\n";
          output += "  </row>\n";
        }
      }
      //output += "  <row>\n    <field name=\"" + rows->last().at(0) + "\">" + rows->last().at(0) + "</field>\n  </row>\n";
      output += "</resultset>\n";
    }
  }
  if (saveToFile) {
    saveOutputToFile(output, "XML Files (*.xml)", settings.value("GeneralSettings/LastXMLFile", "").toString());
    return "";
  } else {
    return output;
  }
}

QString DBMS::outputAsV(QString queryToExecute, bool printRowsInSet, bool saveToFile, bool replaceReturns, bool splitQuery, bool removeHeaders, QString delimiter)
{
  int row = 0;
  int row2 = 0;
  QString output;
  QStringList queryList;
  if (splitQuery)
    queryList = queryToExecute.split(QRegularExpression("\\" + delimiter + "\\s+"), Qt::SkipEmptyParts);
  else
    queryList.append(queryToExecute);

  if (isOpened()) {
    foreach (QString statement, queryList) {
      QList<QStringList> *rows = runQuery(statement, true);

      if (printRowsInSet)
        output += "--------------\n" + statement + "\n--------------\n";

      if (removeHeaders)
        rows->removeAt(0);

      //Print the data
      if (replaceReturns) {
        for (row = 0; row < rows->count() - 1; row++) {
          for (row2 = 0; row2 < rows->at(row).count(); row2++)
            output +=  replaceReturnsAndTabs(rows->at(row).at(row2)) + "\t";
          output += "\n";
        }
      } else {
        for (row = 0; row < rows->count() - 1; row++) {
          for (row2 = 0; row2 < rows->at(row).count(); row2++)
            output +=  QString::fromUtf8(rows->at(row).at(row2).toUtf8()) + "\t";
          output += "\n";
        }
      }
      if (printRowsInSet)
        output += "\n" + rows->last().at(0) + "\n";
    }
  }

  if (saveToFile) {
    saveOutputToFile(output, "Text Files (*.txt)", settings.value("GeneralSettings/LastTextFile", "").toString());
    return "";
  } else {
    return output;
  }
}

QString DBMS::outputAsJ(QString queryToExecute, bool saveToFile, bool replaceReturns, bool splitQuery, bool emitNotificacion, QString delimiter)
{
  QString output;
  int row = 0;
  int row2 = 0;
  QStringList queryList;
  if (splitQuery)
    queryList = queryToExecute.split(QRegularExpression("\\" + delimiter + "\\s+"), Qt::SkipEmptyParts);
  else
    queryList.append(queryToExecute);

  if (isOpened()) {
    foreach (QString statement, queryList) {
      QList<QStringList> *rows = runQuery(statement, true, emitNotificacion);

      //Print the data
      output += "[\n";
      if (replaceReturns) {
        for (row = 1; row < rows->count() - 1; row++) {
          output += "{\n";
          for (row2 = 0; row2 < rows->at(row).count(); row2++) {
            output += "  \"" + replaceReturnsAndTabs(rows->at(0).at(row2)) + "\": ";
            output += "\"" + replaceReturnsAndTabs(rows->at(row).at(row2)) + "\",\n";
          }
          output = output.removeLast();
          output = output.removeLast();
          output += "\n},\n";
        }
        output = output.removeLast();
        output = output.removeLast();
      } else {
        for (row = 1; row < rows->count() - 1; row++) {
          output += "{\n";
          for (row2 = 0; row2 < rows->at(row).count(); row2++) {
            output += "  \"" + replaceReturnsAndTabs(rows->at(0).at(row2)) + "\": ";
            output += "\"" + QString::fromUtf8(rows->at(row).at(row2).toUtf8()) + "\",\n";
          }
          output = output.removeLast();
          output = output.removeLast();
          output += "\n},\n";
        }
        output = output.removeLast();
        output = output.removeLast();
      }
      output += "\n]\n" + rows->last().at(0) + "\n";
    }
  }
  if (saveToFile) {
    saveOutputToFile(output, "Text Files (*.txt)", settings.value("GeneralSettings/LastTextFile", "").toString());
    return "";
  } else {
    return output;
  }
}

QString DBMS::outputAsG(QString queryToExecute, bool saveToFile, bool replaceReturns, bool splitQuery, bool emitNotificacion, QString delimiter)
{
  QString output;
  int maxLength = 0;
  int row = 0;
  int row2 = 0;
  QStringList queryList;
  if (splitQuery)
    queryList = queryToExecute.split(QRegularExpression("\\" + delimiter + "\\s+"), Qt::SkipEmptyParts);
  else
    queryList.append(queryToExecute);

  if (isOpened()) {
    foreach (QString statement, queryList) {
      QList<QStringList> *rows = runQuery(statement, true, emitNotificacion);

      for (row = 0; row < rows->at(0).count(); row++)
        if (maxLength < rows->at(0).at(row).length())
          maxLength = rows->at(0).at(row).length();

      //Print the data
      if (replaceReturns) {
        for (row = 1; row < rows->count() - 1; row++) {
          for (row2 = 0; row2 < rows->at(row).count(); row2++) {
            output += replaceReturnsAndTabs(rows->at(0).at(row2)).rightJustified(maxLength, ' ') + ": ";
            output += replaceReturnsAndTabs(rows->at(row).at(row2)).leftJustified(maxLength, ' ') + "\n";
          }
          output += "\n";
        }

      } else {
        for (row = 1; row < rows->count() - 1; row++) {
          for (row2 = 0; row2 < rows->at(row).count(); row2++) {
            output += replaceReturnsAndTabs(rows->at(0).at(row2)).rightJustified(maxLength, ' ') + ": ";
            output += QString::fromUtf8(rows->at(row).at(row2).toUtf8()).leftJustified(maxLength, ' ') + "\n";
          }
          output += "\n";
        }
      }
      output += "\n" + rows->last().at(0) + "\n";
    }
  }
  if (saveToFile) {
    saveOutputToFile(output, "Text Files (*.txt)", settings.value("GeneralSettings/LastTextFile", "").toString());
    return "";
  } else {
    return output;
  }
}

QString DBMS::errorOnExecution(const QString message, const QString type, const QString statement)
{
  emit statusBarMessage(message, QSystemTrayIcon::Critical, 0);
  emit errorOccurred();
  qWarning() << message << statement;
  errorMessage->showMessage(message, type);
  return tr("Could not execute statement. ") + message;
}

QString DBMS::outputAsVV(QString queryToExecute, bool saveToFile, bool replaceReturns, bool splitQuery, QString delimiter)
{
  return outputAsV(queryToExecute, true, saveToFile, replaceReturns, splitQuery, false, delimiter);
}

int DBMS::getPort()
{
  return port;
}

int DBMS::getConnectionId()
{
  if (isOpened())
    return runQuerySingleColumn("SELECT CONNECTION_ID()").at(0).toInt();
  return -1;
}

QString DBMS::getSocket()
{
  if (isOpened())
    switch(p_DBMSType) {
    case StaticFunctions::MySQL:
      if (getVersion().left(3) == "5.7") {
        return runQuerySingleColumn("SELECT `VARIABLE_VALUE` FROM `performance_schema`.`session_variables` WHERE `VARIABLE_NAME` = 'SOCKET'").at(0).at(0);
      }
      break;
    case StaticFunctions::MariaDB:
      return runQuerySingleColumn("SELECT `VARIABLE_VALUE` FROM `information_schema`.`SESSION_VARIABLES` WHERE `VARIABLE_NAME` = 'SOCKET'").at(0).at(0);
      break;
    case StaticFunctions::Undefined:
    default:
      break;
    }
  return QString();
}

QString DBMS::getUserName()
{
  return p_userName;
}

QString DBMS::getFullUserName()
{
  if (isOpened())
    return runQuerySingleColumn("SELECT CONCAT(CONCAT('`', SUBSTRING_INDEX(CURRENT_USER(), '@', 1), '`'), '@', CONCAT('`', SUBSTRING_INDEX(CURRENT_USER(), '@', -1), '`'))").at(0);
  return QString();
}

QString DBMS::getPassword()
{
  return password;
}

QString DBMS::getHostName()
{
  return hostName;
}

QString DBMS::getSessionStatus(QString filter)
{
  if (isOpened())
    switch(p_DBMSType) {
    case StaticFunctions::MySQL:
    case StaticFunctions::MariaDB:
      if (!filter.isEmpty())
        return outputAsTable("SELECT * FROM " + getSessionStatusTable() + " WHERE `VARIABLE_NAME` LIKE '%" + escapeString(filter) + "%' ORDER BY `VARIABLE_NAME`");
      else
        return outputAsTable("SELECT * FROM " + getSessionStatusTable() + " ORDER BY `VARIABLE_NAME`");
      break;
    case StaticFunctions::Undefined:
    default:
      break;
    }
  return QString();
}

QString DBMS::getSessionlVariables(QString filter)
{
  if (isOpened())
    switch(p_DBMSType) {
    case StaticFunctions::MySQL:
    case StaticFunctions::MariaDB:
      if (!filter.isEmpty())
        return outputAsTable("SELECT * FROM " + getSessionVariablesTable() + " WHERE `VARIABLE_NAME` LIKE '%" + escapeString(filter) + "%' ORDER BY `VARIABLE_NAME`");
      else
        return outputAsTable("SELECT * FROM " + getSessionVariablesTable() + " ORDER BY `VARIABLE_NAME`");
      break;
    case StaticFunctions::Undefined:
    default:
      break;
    }
  return QString();
}

QString DBMS::getGlobalStatus(QString filter)
{
  if (isOpened())
    switch(p_DBMSType) {
    case StaticFunctions::MySQL:
    case StaticFunctions::MariaDB:
      if (!filter.isEmpty())
        return outputAsTable("SELECT * FROM " + getGlobalStatusTable() + " WHERE `VARIABLE_NAME` LIKE '%" + escapeString(filter) + "%' ORDER BY `VARIABLE_NAME`");
      else
        return outputAsTable("SELECT * FROM " + getGlobalStatusTable() + " ORDER BY `VARIABLE_NAME`");
      break;
    case StaticFunctions::Undefined:
    default:
      break;
    }
  return QString();
}

QString DBMS::getGlobalVariables(QString filter)
{
  if (isOpened())
    switch(p_DBMSType) {
    case StaticFunctions::MySQL:
    case StaticFunctions::MariaDB:
      if (!filter.isEmpty())
        return outputAsTable("SELECT * FROM " + getGlobalVariablesTable() + " WHERE `VARIABLE_NAME` LIKE '%" + escapeString(filter) + "%' ORDER BY `VARIABLE_NAME`");
      else
        return outputAsTable("SELECT * FROM " + getGlobalVariablesTable() + " ORDER BY `VARIABLE_NAME`");
      break;
    case StaticFunctions::Undefined:
    default:
      break;
    }
  return QString();
}

QString DBMS::getStatus()
{
  if (isOpened())
    return QString("%1").arg(mysql_stat(mariadbConnection));
  return QString();
}

bool DBMS::executeQuery(QString queryToExecute)
{
  if (query(queryToExecute) > 0)
    return true;
  else
    return false;
}

void DBMS::flushPrivileges()
{
  if (isOpened())
    query("FLUSH PRIVILEGES");
}

void DBMS::flushHosts()
{
  if (isOpened())
    query("FLUSH HOSTS");
}

QList<QStringList> *DBMS::getPrivileges(QString filter)
{
  QList<QStringList> *rows = new QList<QStringList>();
  rows = runQuery("SHOW PRIVILEGES", false, false);
  rows->takeLast();
  if (!filter.isEmpty()) {
    QList<QStringList> *rows2 = new QList<QStringList>();
    for (int counter = 0; counter < rows->count(); counter++) {
      if (rows->at(counter).at(1).contains(filter)) {
        rows2->append(rows->at(counter));
      }
    }
    rows = rows2;
  }
  return rows;
}

QString DBMS::getStringType()
{
  switch(p_DBMSType) {
  case StaticFunctions::MySQL:
    return "MySQL";
  case StaticFunctions::MariaDB:
    return "MariaDB";
    break;
  case StaticFunctions::Undefined:
    return "--";
  default:
    break;
  }
  return QString();
}

QString DBMS::replaceReturnsAndTabs(QString string)
{
  static QRegularExpression re("(\\r|\\n|\\t)");
  return QString::fromUtf8(string.toUtf8()).replace(re, " ");
}

QString DBMS::getConnectionString()
{
  //MySQL:root@10.100.20.214:3306/Drupal *Password
  return QString("%1:%2@%3:%4/%5").arg("MariaDB", getUserName(), getHostName(), QString::number(getPort()), getDatabase());
}

QString DBMS::getfailedQueries()
{
  QString output;
  int row = 0;
  int row2 = 0;
  QList<int> maxWidthList;

  //QList<QStringList> *failedQueries = failedQueries;

  //Find the maximum values for the width of the columns
  for (row = 0; row < failedQueries->at(0).count(); row++)
    maxWidthList.append(1);
  for (row = 0; row < failedQueries->count(); row++)
    for (row2 = 0; row2 < failedQueries->at(row).count(); row2++)
      if (QString(failedQueries->at(row).at(row2)).length() > maxWidthList.at(row2))
        maxWidthList.replace(row2, QString(failedQueries->at(row).at(row2)).length());

  //Prints the headres and theirs borders
  output += "+";
  for (row = 0; row < failedQueries->at(0).count(); row++)
    output +=  QString('-').repeated(maxWidthList.at(row) + 2) + "+";
  output += "\n|";
  for (row = 0; row < failedQueries->at(0).count(); row++)
    output +=  " " + replaceReturnsAndTabs(failedQueries->at(0).at(row)).leftJustified(maxWidthList.at(row), ' ') + " |";
  output += "\n+";
  for (row = 0; row < failedQueries->at(0).count(); row++)
    output +=  QString('-').repeated(maxWidthList.at(row) + 2) + "+";

  //Print the data
  for (row = 1; row < failedQueries->count(); row++) {
    output += "\n|";
    for (row2 = 0; row2 < failedQueries->at(row).count(); row2++)
      output +=  " " + replaceReturnsAndTabs(failedQueries->at(row).at(row2)).leftJustified(maxWidthList.at(row2), ' ') + " |";
  }

  //Print the botton border
  output += "\n+";
  for (row = 0; row < failedQueries->at(0).count(); row++)
    output +=  QString('-').repeated(maxWidthList.at(row) + 2) + "+";

  output += "\n";


  return output;
}

void DBMS::logApplicationStarted()
{
  logExecutedQueries(tr("Application started"));
}

void DBMS::logStatement(QString statement, QString result)
{
  logExecutedQueries(statement, result);
}

QSqlTableModel *DBMS::sqliteTableModel()
{
  QSqlTableModel *sqliteModel = new QSqlTableModel;
  if (queryLogEnabled()) {
    sqliteModel->setTable("executedqueries");
    sqliteModel->setEditStrategy(QSqlTableModel::OnFieldChange);
    sqliteModel->select();
    sqliteModel->setHeaderData(0, Qt::Horizontal, tr("Line Number"));
    sqliteModel->setHeaderData(1, Qt::Horizontal, tr("Session Id"));
    sqliteModel->setHeaderData(2, Qt::Horizontal, tr("Date"));
    sqliteModel->setHeaderData(3, Qt::Horizontal, tr("Connection"));
    sqliteModel->setHeaderData(4, Qt::Horizontal, tr("Query"));
  }
  return sqliteModel;
}

QSqlQueryModel *DBMS::sqliteFilterQueryModel()
{
  QSqlQueryModel *sqlQuery = new QSqlQueryModel;
  if (queryLogEnabled())
    sqlQuery->setQuery("SELECT DISTINCT sessionid FROM executedqueries ORDER BY date DESC");
  return sqlQuery;
}

void DBMS::clearSQLiteQueryLog()
{
  if (queryLogEnabled())
    dQueryLogger->clearSQLiteQueryLog();
}

void DBMS::setCharsetAndCollation(QString charset, QString collation)
{
  Q_UNUSED(collation);
  if (isOpened())
    setCharacterSet(charset);
}

void DBMS::logExecutedQueries(QString query, QString result)
{
  if (queryLogEnabled())
    dQueryLogger->logExecutedQueries(query, result);
}

void DBMS::insertOnExecutedQueries(QString sessionid, QString connection, QString query, QString result)
{
  if (queryLogEnabled())
    dQueryLogger->insertOnExecutedQueries(sessionid, connection, query, result);
}

MYSQL *DBMS::getMariadbConnection()
{
  return this->mariadbConnection;
}

bool DBMS::dummyExecution(QString statement)
{
  unsigned int errorOccurred;
  transaction()->beginTransacction();
  runQuery(statement);
  errorOccurred = mysql_errno(mariadbConnection);
  transaction()->rollbackTransacction();
  return errorOccurred != 0 ? false : true;
}

QString DBMS::escapeString(const QString &input)
{
  if (!isOpened() || !mariadbConnection) {
      qWarning() << tr("Attempt to escape string without database connection.");
      return input;
  }
  QByteArray src = input.toUtf8();
  QByteArray dest;
  dest.resize(src.size() * 2 + 1);
  unsigned long len = mysql_real_escape_string(mariadbConnection, dest.data(), src.data(), src.size());
  dest.resize(len);
  return QString::fromUtf8(dest);
}

QString DBMS::getVersion()
{
  if (isOpened())
    return runQuerySingleColumn("SELECT VERSION()").at(0);
  return QString();
}

int unsigned DBMS::getMayorVersion()
{
  if (isOpened())
    return QString(runQuerySingleColumn("SELECT SUBSTRING_INDEX(SUBSTRING_INDEX(VERSION(), '-', 1), '.', 1)").at(0)).toInt();
  return 0;
}

int unsigned DBMS::getMinorVersion()
{
  if (isOpened())
    return QString(runQuerySingleColumn("SELECT SUBSTRING_INDEX(SUBSTRING_INDEX(SUBSTRING_INDEX(VERSION(), '-', 1), '.', 2), '.', -1)").at(0)).toInt();
  return 0;
}

int unsigned DBMS::getMicroVersion()
{
  if (isOpened())
    return QString(runQuerySingleColumn("SELECT SUBSTRING_INDEX(SUBSTRING_INDEX(VERSION(), '-', 1), '.', -1)").at(0)).toInt();
  return 0;
}

QString DBMS::getDatabase()
{
  return p_database;
}

bool DBMS::changeDatabase(QString database)
{
  if (isOpened()) {
    if (query(QString("USE " + StaticFunctions::quoteSymbol(database))) == 0) {
      setDatabase(database);
      return true;
    } else {
      return false;
    }
  }
  emit databaseChanged();
  return false;
}

QString DBMS::getTriggeredefinition(QString trigger, QString database)
{
  if (database.isEmpty())
    database = getDatabase();
  if (isOpened())
    return runQuery("SHOW CREATE TRIGGER " + StaticFunctions::quoteSymbol(database) + "." + StaticFunctions::quoteSymbol(trigger))->at(0).at(2);
  return QString();
}

QStringList DBMS::getEngines()
{
  if (isOpened())
    return runQuerySingleColumn("SELECT `ENGINE` FROM `information_schema`.`ENGINES` ORDER BY `ENGINE`");
  return QStringList();
}

QStringList DBMS::getCollations()
{
  if (isOpened())
    return runQuerySingleColumn("SELECT DISTINCT `COLLATION_NAME` FROM `information_schema`.`COLLATIONS` ORDER BY `COLLATION_NAME`");
  return QStringList();
}

QList<QStringList> *DBMS::getCollationsApplicability()
{
  if (isOpened())
    return runQuery("SELECT `COLLATION_NAME`, CHARACTER_SET_NAME FROM `information_schema`.`COLLATIONS` ORDER BY `COLLATION_NAME`");
  return new QList<QStringList>();
}

void DBMS::setKeyFile(QString keyFile)
{
  this->p_clientKey = keyFile;
}

QString DBMS::getKeyFile()
{
  return this->p_clientKey;
}

void DBMS::setCertFile(QString certFile)
{
  this->p_clientCert = certFile;
}

QString DBMS::getCertFile()
{
  return this->p_clientCert;
}

void DBMS::setUseSSL(bool useSSL)
{
  this->p_useSSL = useSSL;
}

bool DBMS::getUseSSL()
{
  return this->p_useSSL;
}

int DBMS::ping()
{
  if (isOpened())
    return mysql_ping(mariadbConnection);
  return -1;
}

StaticFunctions::dbmsTypes DBMS::getDBMSType()
{
  return p_DBMSType;
}

void DBMS::setDBMSType(StaticFunctions::dbmsTypes type)
{
  qApp->setProperty("DBMSType", type);
  p_DBMSType = type;
}

QStringList DBMS::getCharsets()
{
  if (isOpened())
    return runQuerySingleColumn("SELECT DISTINCT `CHARACTER_SET_NAME` FROM `information_schema`.`COLLATIONS` ORDER BY `CHARACTER_SET_NAME`");
  return QStringList();
}

QList<QStringList> *DBMS::getCharacterSets()
{
  if (isOpened()) {
    QList<QStringList> *rows = runQuery("SELECT `CHARACTER_SET_NAME`, `DESCRIPTION` FROM `information_schema`.`CHARACTER_SETS` ORDER BY `CHARACTER_SET_NAME`");
    rows->takeLast();
    return rows;
  }
  return new QList<QStringList>;
}

QString DBMS::getCharacterSet()
{
  return p_charset;
}

void DBMS::setCharacterSet(QString charset)
{
  if (isOpened()) {
    if (mysql_set_character_set(mariadbConnection, charset.toUtf8()) != 0)
      errorOnExecution(tr("Could not change character set to: %1").arg(charset), "SetCharacterSet", "mysql_set_character_set(mariadbConnection, charset.toUtf8())");
    settings.setValue("DBMS/CharacterSet", charset);
  }
  p_charset = charset;
}

Table *DBMS::table(QString tableName, QString database)
{
  return new Table(this, tableName, database);
}

Database *DBMS::database(QString databaseName)
{
  return new Database(this, databaseName.isEmpty() ? getDatabase() : databaseName);
}

DatabaseEvent *DBMS::databaseEvent(QString eventName, QString database)
{
  return new DatabaseEvent(this, eventName, database);
}

Processes *DBMS::processes()
{
  return new Processes(this);
}

Replication *DBMS::replication()
{
  return new Replication(this);
}

User *DBMS::user(QString userName)
{
  return new User(this, userName);
}

void DBMS::errorMessageAcceptedSlot()
{
  emit errorMessageAccepted();
}

void DBMS::checkIfReconnected()
{
  if (isOpened()) {
    QList result = runQuerySingleColumn("SELECT CONNECTION_ID()");
    if (p_connectionId != (result.count() > 0 ? result.at(0).toULong() : 0))
      emit reconnectionPerformed();
  }
}

View *DBMS::view(QString viewName, QString database)
{
  return new View(this, viewName, database);
}

Trigger *DBMS::trigger(QString triggerName, QString database)
{
  return new Trigger(this, triggerName, database);
}

Routine *DBMS::routine(QString routineName, QString database)
{
  return new Routine(this, routineName, database);
}

Transaction *DBMS::transaction()
{
  return new Transaction(this);
}

QString DBMS::lastErrorNumber()
{
  if (isOpened())
    return QString("%1").arg(mysql_errno(mariadbConnection));
  return tr("No error code provided.");
}

Triggers *DBMS::triggers()
{
  return new Triggers(this);
}

Tables *DBMS::tables()
{
  return new Tables(this);
}

Views *DBMS::views()
{
  return new Views(this);
}

Events *DBMS::events()
{
  return new Events(this);
}

Functions *DBMS::functions()
{
  return new Functions(this);
}

Procedures *DBMS::procedures()
{
  return new Procedures(this);
}

Databases *DBMS::databases()
{
  return new Databases(this);
}

/***************************************************************************************************************/

Database::Database(DBMS *serverConnection, QString databaseName)
{
  this->serverConnection = serverConnection;
  if (databaseName.isEmpty())
    this->databaseName = serverConnection->getDatabase();
  else
    this->databaseName = databaseName;
}

bool Database::create(QString collation)
{
  if (collation.isEmpty())
    collation = "utf8mb4_general_ci";
  return serverConnection->executeQuery(QString("CREATE DATABASE %1 COLLATE %2").arg(formalName(), collation));
}

QString Database::formalName()
{
  return StaticFunctions::quoteSymbol(databaseName);
}

bool Database::drop()
{
  return serverConnection->executeQuery(QString("DROP DATABASE %1").arg(formalName()));
}

QStringList Database::getTables()
{
  return serverConnection->runQuerySingleColumn("SELECT `TABLE_NAME` FROM `information_schema`.`TABLES` WHERE `TABLE_TYPE` IN ('BASE TABLE', 'SYSTEM VERSIONED') AND `TABLE_SCHEMA` = '" + databaseName + "' ORDER BY `TABLE_NAME`");
}

QStringList Database::getEvents()
{
  return serverConnection->runQuerySingleColumn("SELECT `EVENT_NAME` FROM `information_schema`.`EVENTS` WHERE `EVENT_SCHEMA` = '" + databaseName + "' ORDER BY `EVENT_NAME`");
}

QStringList Database::getViews()
{
  return serverConnection->runQuerySingleColumn("SELECT `TABLE_NAME` FROM `information_schema`.`TABLES` WHERE `TABLE_TYPE` = 'VIEW' AND `TABLE_SCHEMA` = '" + databaseName + "' ORDER BY `TABLE_NAME`");
}

QStringList Database::getTriggers()
{
  return serverConnection->runQuerySingleColumn("SELECT `TRIGGER_NAME` FROM `information_schema`.`TRIGGERS` WHERE `TRIGGER_SCHEMA` = '" + databaseName + "' ORDER BY `TRIGGER_NAME`");
}

QStringList Database::getRoutines()
{
  return serverConnection->runQuerySingleColumn("SELECT `ROUTINE_NAME` FROM `information_schema`.`ROUTINES` WHERE `ROUTINE_SCHEMA` = '" + databaseName + "' ORDER BY `ROUTINE_NAME`");
}

unsigned long Database::tableCount()
{
  return serverConnection->runQuery("SELECT COUNT(*) FROM `information_schema`.`TABLES` WHERE `TABLE_SCHEMA` = '" + databaseName + "'")->at(0).at(0).toULong();
}

QStringList Database::info()
{
  return QStringList()
      << tr("Database: %1").arg(databaseName)
      << tr("Total of tables: %1").arg(serverConnection->runQuerySingleColumn("SELECT COUNT(*) FROM `information_schema`.`TABLES` WHERE `TABLE_SCHEMA` = '" + databaseName + "'").at(0))
      << tr("Total of views: %1").arg(serverConnection->runQuerySingleColumn("SELECT COUNT(*) FROM `information_schema`.`VIEWS` WHERE `TABLE_SCHEMA` = '" + databaseName + "'").at(0))
      << tr("Total of triggers: %1").arg(serverConnection->runQuerySingleColumn("SELECT COUNT(*) FROM `information_schema`.`TRIGGERS` WHERE `TRIGGER_SCHEMA` = '" + databaseName + "'").at(0))
      << tr("Total of rutines: %1").arg(serverConnection->runQuerySingleColumn("SELECT COUNT(*) FROM `information_schema`.`ROUTINES` WHERE `ROUTINE_SCHEMA` = '" + databaseName + "'").at(0))
      << tr("Total of events: %1").arg(serverConnection->runQuerySingleColumn("SELECT COUNT(*) FROM `information_schema`.`EVENTS` WHERE `EVENT_SCHEMA` = '" + databaseName + "'").at(0))
      << tr("Tables size: %1").arg(serverConnection->runQuerySingleColumn("SELECT CONCAT(FORMAT((SUM(`DATA_LENGTH`) / 1024 / 1024), 2), ' MB.') AS `Data` FROM `information_schema`.`TABLES` WHERE `TABLE_SCHEMA` = '" + databaseName + "'").at(0))
      << tr("Idexes: %1").arg(serverConnection->runQuerySingleColumn("SELECT CONCAT(FORMAT((SUM(`INDEX_LENGTH`) / 1024 / 1024), 2), ' MB.') AS `Indexes` FROM `information_schema`.`TABLES` WHERE `TABLE_SCHEMA` = '" + databaseName + "'").at(0))
         ;
}

QStringList Database::getLocalTables()
{
  return serverConnection->runQuerySingleColumn("SELECT `TABLE_NAME` FROM `information_schema`.`TABLES` WHERE `TABLE_TYPE` IN ('BASE TABLE', 'SYSTEM VERSIONED') AND `ENGINE` NOT IN ('FEDERATED', 'MEMORY') AND `TABLE_SCHEMA` = '" + databaseName + "' ORDER BY `TABLE_NAME`");
}

QString Database::getDefinition()
{
  return serverConnection->runQuery("SHOW CREATE DATABASE " + databaseName)->at(0).at(1);
}

/***************************************************************************************************************/

Table::Table(DBMS *serverConnection, QString tableName, QString database)
{
  this->serverConnection = serverConnection;
  this->tableName = StaticFunctions::quoteSymbol(tableName);
  if (database.isEmpty())
    this->database = StaticFunctions::quoteSymbol(serverConnection->getDatabase());
  else
    this->database = StaticFunctions::quoteSymbol(database);
}

unsigned long Table::getChecksum()
{
  return serverConnection->runQuery("CHECKSUM TABLE " + formalName())->at(0).at(1).toULong();
}

unsigned long Table::getRowCount()
{
  return serverConnection->runQuery("SELECT COUNT(*) FROM " + formalName())->at(0).at(0).toULong();
}

unsigned long Table::getDataLength()
{
  return serverConnection->runQuery("SELECT `DATA_LENGTH` FROM `information_schema`.`TABLES` WHERE `TABLE_SCHEMA` = '"
                                    + StaticFunctions::unquoteSymbol(database)
                                    + "' AND `TABLE_NAME` = '"
                                    + StaticFunctions::unquoteSymbol(tableName)
                                    + "'")->at(0).at(0).toULong();
}

bool Table::renameTable(QString newName)
{
  return serverConnection->executeQuery("RENAME TABLE " + formalName()
                                        + " TO " + StaticFunctions::quoteSymbol(database)
                                        + "." + StaticFunctions::quoteSymbol(newName));
  this->tableName = StaticFunctions::quoteSymbol(newName);
}

bool Table::changeEngine(QString newEngine)
{
  return serverConnection->executeQuery("ALTER TABLE " + formalName() + " ENGINE " + newEngine);
}

bool Table::changeComment(QString newComment)
{
  return serverConnection->executeQuery("ALTER TABLE " + formalName() + " COMMENT '" + newComment + "'");
}

bool Table::changeCollation(QString newCollation)
{
  return serverConnection->executeQuery("ALTER TABLE " + formalName() + " COLLATE " + newCollation);
}

bool Table::drop()
{
  return serverConnection->executeQuery("DROP TABLE IF EXISTS " + formalName());
}

QString Table::formalName()
{
  return database + "." + tableName;
}

QString Table::getDefinition(bool returnFormalName)
{
  if (returnFormalName)
    return QString(serverConnection->runQuery("SHOW CREATE TABLE " + formalName())->at(0).at(1))
        .replace("CREATE TABLE " + tableName + " (", "CREATE TABLE " + formalName() + " (");
  else
    return QString(serverConnection->runQuery("SHOW CREATE TABLE " + formalName())->at(0).at(1));
}

QStringList Table::getFields()
{
  return serverConnection->runQuerySingleColumn("SHOW COLUMNS FROM " + formalName());
}

bool Table::dropIndex(QString indexName)
{
  return serverConnection->runQuery("DROP INDEX " + StaticFunctions::quoteSymbol(indexName) + " ON " + formalName());
}

QList<QStringList>* Table::getIndexes()
{
  return serverConnection->runQuery("SHOW INDEX FROM " + formalName());
}

/***************************************************************************************************************/

View::View(DBMS *serverConnection, QString viewName, QString database)
{
  this->serverConnection = serverConnection;
  this->viewName = StaticFunctions::quoteSymbol(viewName);;
  if (database.isEmpty())
    this->database = StaticFunctions::quoteSymbol(serverConnection->getDatabase());
  else
    this->database = StaticFunctions::quoteSymbol(database);
}

bool View::drop()
{
  return serverConnection->executeQuery("DROP VIEW IF EXISTS " + formalName());
}

QString View::formalName()
{
  return database + "." + viewName;
}

QString View::getDefinition()
{
  return serverConnection->runQuery("SHOW CREATE VIEW " + formalName())->at(0).at(1);
}

/***************************************************************************************************************/

Routine::Routine(DBMS *serverConnection, QString routineName, QString database)
{
  this->serverConnection = serverConnection;
  this->routineName = StaticFunctions::quoteSymbol(routineName);
  if (database.isEmpty())
    this->database = StaticFunctions::quoteSymbol(serverConnection->getDatabase());
  else
    this->database = StaticFunctions::quoteSymbol(database);
}

bool Routine::drop()
{
  return serverConnection->executeQuery(QString("DROP %1 IF EXIST %2").arg(routineType(), formalName()));
}

QString Routine::formalName()
{
  return database + "." + routineName;
}

QString Routine::routineType()
{
  if (serverConnection->runQuerySingleColumn("SELECT `ROUTINE_TYPE` FROM `information_schema`.`ROUTINES` WHERE `ROUTINE_NAME` = '"
                                             + routineName.mid(1, routineName.length() - 2) + "' AND `ROUTINE_SCHEMA` = '"
                                             + database.mid(1, database.length() - 2) + "'").at(0) == "FUNCTION")
    return "FUNCTION";
  else
    return "PROCEDURE";
}

QString Routine::getDefinition()
{
  return serverConnection->runQuery("SHOW CREATE " + routineType() + " " + formalName())->at(0).at(2);
}

/***************************************************************************************************************/

DatabaseEvent::DatabaseEvent(DBMS *serverConnection, QString eventName, QString database)
{
  this->serverConnection = serverConnection;
  this->eventName = StaticFunctions::quoteSymbol(eventName);
  if (database.isEmpty())
    this->database = StaticFunctions::quoteSymbol(serverConnection->getDatabase());
  else
    this->database = StaticFunctions::quoteSymbol(database);
}

bool DatabaseEvent::drop()
{
  return serverConnection->runQuery("DROP EVENT " + formalName());
}

QString DatabaseEvent::formalName()
{
  return database + "." + eventName;
}

QString DatabaseEvent::getDefinition()
{
  return serverConnection->runQuery("SHOW CREATE EVENT " + formalName())->at(0).at(3);
}

/***************************************************************************************************************/

Processes::Processes(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
}

QList<QStringList> *Processes::getProcessList(bool useTable, QString userName)
{
  switch(serverConnection->getDBMSType()) {
  case StaticFunctions::MySQL:
  case StaticFunctions::MariaDB:
    if (useTable)
      result = serverConnection->runQuery("SELECT * FROM `information_schema`.`PROCESSLIST`"
                                          + (!(userName == tr("All users")) ? " WHERE `User` = '" + userName + "'" : "")
                                          , false, false);
    else
      result = serverConnection->runQuery("SHOW FULL PROCESSLIST", false, false);
    break;
  case StaticFunctions::Undefined:
  default:
    break;
  }
  result->takeLast(); //Remove the "Affected rows" line.
  return result;
}

QStringList Processes::getHeaderList(bool useTable)
{
  switch(serverConnection->getDBMSType()) {
  case StaticFunctions::MySQL:
  case StaticFunctions::MariaDB:
    if (useTable)
      return serverConnection->runQuery("SELECT * FROM `information_schema`.`PROCESSLIST`", true, false)->takeFirst();
    else
      return serverConnection->runQuery("SHOW FULL PROCESSLIST", true, false)->takeFirst();
    break;
  case StaticFunctions::Undefined:
  default:
    break;
  }
  return QStringList();
}

void Processes::killThread(long long thread)
{
  serverConnection->runQuery(QString("KILL CONNECTION %1").arg(thread));
}

void Processes::killQuery(long long thread)
{
  serverConnection->runQuery(QString("KILL QUERY %1").arg(thread));
}

void Processes::killIdleThreads(unsigned int limit)
{
  result = serverConnection->runQuery(QString("SELECT `ID` FROM `information_schema`.`PROCESSLIST` WHERE `TIME` > %1 AND `COMMAND` NOT IN ('Daemon', 'Binlog Dump') AND `INFO` IS NULL AND `USER` <> 'federated'").arg(limit));
  result->removeLast();
  for (int row = 0; row < result->count(); row++)
    serverConnection->runQuery(QString("KILL CONNECTION %1").arg(result->at(row).at(0)));
}

void Processes::killBusyThreads(unsigned int limit)
{
  result = serverConnection->runQuery(QString("SELECT `ID` FROM `information_schema`.`PROCESSLIST` WHERE `TIME` > %1 AND `COMMAND` NOT IN ('Daemon', 'Binlog Dump') AND `USER` <> 'federated'").arg(limit));
  result->removeLast();
  for (int row = 0; row < result->count(); row++)
    serverConnection->runQuery(QString("KILL CONNECTION %1").arg(result->at(row).at(0)));
}

/***************************************************************************************************************/

Replication::Replication(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
}

void Replication::changeDefaultMasterConnection(QString masterConnectionName)
{
  if (serverConnection->isOpened())
    serverConnection->runQuery("SET `default_master_connection` := '" + masterConnectionName + "'");
}

QString Replication::getSlaveStatus()
{
  if (serverConnection->isOpened())
    switch(serverConnection->getDBMSType()) {
    case StaticFunctions::MySQL:
      return serverConnection->outputAsG("SHOW SLAVE STATUS", false, true, true, false);
      break;
    case StaticFunctions::MariaDB:
      return serverConnection->outputAsG("SHOW ALL SLAVES STATUS", false, true, true, false);
      break;
    case StaticFunctions::Undefined:
    default:
      break;
    }
  return QString();
}

QString Replication::getMasterStatus()
{
  return serverConnection->outputAsTable("SHOW MASTER STATUS", false, false, true, true, false);
}

void Replication::skipErrors(unsigned int count)
{
  serverConnection->executeQuery(QString("STOP SLAVE"));
  serverConnection->executeQuery(QString("SET GLOBAL SQL_SLAVE_SKIP_COUNTER = %1").arg(count));
  serverConnection->executeQuery(QString("START SLAVE;"));
}

void Replication::stopSlave()
{
  if (serverConnection->isOpened())
    serverConnection->executeQuery("STOP SLAVE");
}

void Replication::startSlave()
{
  if (serverConnection->isOpened())
    serverConnection->executeQuery("START SLAVE");
}

void Replication::rebootSlave()
{
  if (serverConnection->isOpened()) {
    serverConnection->executeQuery("STOP SLAVE");
    serverConnection->executeQuery("START SLAVE");
  }
}

void Replication::resetSlave()
{
  if (serverConnection->isOpened())
    serverConnection->executeQuery("RESET SLAVE");
}

void Replication::purgeBinaryLogs()
{
  if (serverConnection->isOpened())
    serverConnection->executeQuery("PURGE BINARY LOGS BEFORE NOW()");
}

void Replication::flushRelayLogs()
{
  if (serverConnection->isOpened())
    serverConnection->executeQuery("FLUSH REALY LOGS");
}

void Replication::stopAllSlaves()
{
  if (serverConnection->isOpened())
    serverConnection->executeQuery("STOP ALL SLAVES");
}

void Replication::startAllSlaves()
{
  if (serverConnection->isOpened())
    serverConnection->executeQuery("START ALL SLAVES");
}

QString Replication::getSlaveStatus(QString connectionName)
{
  if (serverConnection->isOpened())
    return serverConnection->outputAsG("SHOW SLAVE '" + connectionName + "' STATUS", false, true, true, false);
  return QString();
}

QStringList Replication::getSlavesNames()
{
  if (serverConnection->isOpened())
    return serverConnection->runQuerySingleColumn("SHOW ALL SLAVES STATUS", false);
  return QStringList();
}

void Replication::stopSlave(QString connectionName)
{
  if (serverConnection->isOpened())
    serverConnection->executeQuery("STOP SLAVE '" + connectionName + "'");
}

void Replication::startSlave(QString connectionName)
{
  if (serverConnection->isOpened())
    switch(serverConnection->getDBMSType()) {
    case StaticFunctions::MySQL:
      break;
    case StaticFunctions::MariaDB:
      serverConnection->executeQuery("START SLAVE '" + connectionName + "'");
      break;
    case StaticFunctions::Undefined:
    default:
      break;
    }
}

void Replication::rebootSlave(QString connectionName)
{
  if (serverConnection->isOpened()) {
    serverConnection->executeQuery("STOP SLAVE '" + connectionName + "'");
    serverConnection->executeQuery("START SLAVE '" + connectionName + "'");
  }
}

void Replication::resetSlave(QString connectionName)
{
  if (serverConnection->isOpened())
    serverConnection->executeQuery("RESET SLAVE '" + connectionName + "'");
}

/***************************************************************************************************************/

Transaction::Transaction(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
}

void Transaction::beginTransacction()
{
  if (serverConnection->isOpened()) {
    serverConnection->executeQuery("START TRANSACTION");
    emit serverConnection->statusBarMessage(tr("Transaction started"), QSystemTrayIcon::Information, 0);
  }
}

void Transaction::commitTransacction()
{
  if (serverConnection->isOpened()) {
    serverConnection->executeQuery("COMMIT");
    emit serverConnection->statusBarMessage(tr("Transaction commited"), QSystemTrayIcon::Information, 0);
  }
}

void Transaction::rollbackTransacction()
{
  if (serverConnection->isOpened()) {
    serverConnection->executeQuery("ROLLBACK");
    emit serverConnection->statusBarMessage(tr("Transaction rollbacked"), QSystemTrayIcon::Information, 0);
  }
}

/*******************************************************************************************/

Triggers::Triggers(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
}

QStringList Triggers::list(QString databaseName)
{
  if (databaseName.isEmpty())
    databaseName = serverConnection->getDatabase();
  return serverConnection->runQuerySingleColumn("SELECT CONCAT('`', `TRIGGER_SCHEMA`, '`.`', `TRIGGER_NAME`, '`')  FROM `information_schema`.`TRIGGERS` WHERE `TRIGGER_SCHEMA` = '" + databaseName + "'");
}

QString Triggers::getDefinition(QString formalTriggerName)
{
  return serverConnection->runQuery("SHOW CREATE TRIGGER  " + formalTriggerName)->at(0).at(2);
}

/*******************************************************************************************/

Tables::Tables(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
}

QStringList Tables::list(QString databaseName)
{
  if (databaseName.isEmpty())
    databaseName = serverConnection->getDatabase();
  return serverConnection->runQuerySingleColumn("SELECT CONCAT('`', `TABLE_SCHEMA`, '`.`', `TABLE_NAME`, '`') FROM `information_schema`.`TABLES` WHERE `TABLE_SCHEMA` = '" + databaseName + "' AND `TABLE_TYPE` IN ('BASE TABLE', 'SYSTEM VERSIONED')");
}

QString Tables::getDefinition(QString formalTableName)
{
  return serverConnection->runQuery("SHOW CREATE TABLE  " + formalTableName)->at(0).at(1);
}

QStringList Tables::getTableDataToInsert(QString formalTableName, bool returnFormalName)
{
  /*
   *  This function is good for small tables since all the data is stored in memory.
   */
  QStringList output;
  QStringList tableFormalName =  formalTableName.split(".");
  QString table = tableFormalName.at(1);
  table = table.mid(1, table.length() - 2);
  QString database = tableFormalName.at(0);
  database = database.mid(1, database.length() - 2);
  QString fields;
  QList<QStringList> *rows;
  QList<QStringList> *rows2;
  QString rowTMP;
  QString rowData;
  QString formatedColumns;
  static QRegularExpression regularExpression("(BINARY|BLOB)", QRegularExpression::CaseInsensitiveOption);
  rows2 = serverConnection->runQuery("SELECT `COLUMN_NAME`, `DATA_TYPE` FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA` = '" + database + "' AND `TABLE_NAME` = '" + table + "' ORDER BY `ORDINAL_POSITION` ASC");
  for (int row = 0; row < rows2->count() - 1; row++) {
    fields += "`" + rows2->at(row).at(0) + "`, ";
    if (QString(rows2->at(row).at(1)).contains(regularExpression)) {
      formatedColumns += "HEX(`" + rows2->at(row).at(0) + "`), ";
    } else {
      formatedColumns += "`" + rows2->at(row).at(0) + "`, ";
    }
  }
  fields = fields.mid(0, fields.length() - 2);
  formatedColumns = formatedColumns.mid(0, formatedColumns.length() - 2);
  rowData = QString("INSERT INTO " + (returnFormalName ? formalTableName : "`" + table + "`") + " (" + fields + ") VALUES (");
  rows = serverConnection->runQuery("SELECT " + formatedColumns + " FROM " + formalTableName);

  for (int row = 0; row < rows->count() - 1; row++) {
    for (int row2 = 0; row2 < rows->at(row).count(); row2++) {
      rowTMP += (QString(rows2->at(row2).at(1)).contains(regularExpression) ? "X'" : "'") + QString("%1").arg(rows->at(row).at(row2)).replace("\'", "\\'") + "', ";
    }
    output.append(rowData + rowTMP.mid(0, rowTMP.length() - 2) + ")");
    rowTMP = "";
  }
  return output;
}

/*******************************************************************************************/

Views::Views(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
}

QStringList Views::list(QString databaseName)
{
  if (databaseName.isEmpty())
    databaseName = serverConnection->getDatabase();
  //`information_schema`.`VIEWS` does not have the system views
  return serverConnection->runQuerySingleColumn("SELECT CONCAT('`', `TABLE_SCHEMA`, '`.`', `TABLE_NAME`, '`') FROM `information_schema`.`TABLES` WHERE `TABLE_TYPE` IN ('VIEW', 'SYSTEM VIEW') AND `TABLE_SCHEMA` = '" + databaseName + "'");
}

QString Views::getDefinition(QString formalViewName)
{
  return serverConnection->runQuery("SHOW CREATE VIEW  " + formalViewName)->at(0).at(1);
}

/*******************************************************************************************/

Events::Events(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
}

QStringList Events::list(QString databaseName)
{
  if (databaseName.isEmpty())
    databaseName = serverConnection->getDatabase();
  return serverConnection->runQuerySingleColumn("SELECT CONCAT('`', `EVENT_SCHEMA`, '`.`', `EVENT_NAME`, '`') FROM `information_schema`.`EVENTS` WHERE `EVENT_SCHEMA` = '" + databaseName + "'");
}

QString Events::getDefinition(QString formalEventName)
{
  return serverConnection->runQuery("SHOW CREATE EVENT  " + formalEventName)->at(0).at(3);
}

/*******************************************************************************************/

Functions::Functions(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
}

QStringList Functions::list(QString databaseName)
{
  if (databaseName.isEmpty())
    databaseName = serverConnection->getDatabase();
  return serverConnection->runQuerySingleColumn("SELECT CONCAT('`', `ROUTINE_SCHEMA`, '`.`', `ROUTINE_NAME`, '`') FROM `information_schema`.`ROUTINES` WHERE `ROUTINE_SCHEMA` = '" + databaseName + "' AND `ROUTINE_TYPE` = 'FUNCTION'");
}

QString Functions::getDefinition(QString formalFunctionName)
{
  return serverConnection->runQuery("SHOW CREATE FUNCTION  " + formalFunctionName)->at(0).at(2);
}

/*******************************************************************************************/

Procedures::Procedures(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
}

QStringList Procedures::list(QString databaseName)
{
  if (databaseName.isEmpty())
    databaseName = serverConnection->getDatabase();
  return serverConnection->runQuerySingleColumn("SELECT CONCAT('`', `ROUTINE_SCHEMA`, '`.`', `ROUTINE_NAME`, '`') FROM `information_schema`.`ROUTINES` WHERE `ROUTINE_SCHEMA` = '" + databaseName + "' AND `ROUTINE_TYPE` = 'PROCEDURE'");
}

QString Procedures::getDefinition(QString formalEventName)
{
  return serverConnection->runQuery("SHOW CREATE PROCEDURE  " + formalEventName)->at(0).at(2);
}

/*******************************************************************************************/

Databases::Databases(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
}

QStringList Databases::list()
{
  return serverConnection->runQuerySingleColumn("SHOW DATABASES");
}

QString Databases::getDefinition(QString formalEventName)
{
  return serverConnection->runQuery("SHOW CREATE DATABASE  " + formalEventName)->at(0).at(1);
}

void ThreadedQueryExecution::run()
{
  DBMS *serverConnection = new DBMS(false);
  serverConnection->setConnectionParameters(serverConnectionParameters);
  if (!serverConnection->open()) {
    emit errorOccurred(serverConnection->lastError());
    return;
  }
  if (this->query.isEmpty()) {
    serverConnection->close();
    return;
  }
  static QRegularExpression re("(ALTER|CHANGE|CREATE|DELETE|DROP|GRANT|LOAD|RENAME|SET|START|STOP|TRUNCATE|UPDATE)", QRegularExpression::CaseInsensitiveOption);
  if (this->query.contains(re)
      && onlyExecuteSafeStatements) {
    emit stringResultReady(tr("Could not execute statement on safe mode."));
  }
  if (this->outputType == "-t" || this->outputType == "-A" || this->outputType == "-TXT") {
    emit stringResultReady(serverConnection->outputAsTable(query, false, saveToFile, showNewLines, true, true, ";"));
  }
  if (this->outputType == "-x") {
    emit stringResultReady(serverConnection->outputAsV(query, false, saveToFile, showNewLines, true, true, ";"));
  }
  if (this->outputType == "-v") {
    emit stringResultReady(serverConnection->outputAsV(query, false, saveToFile, showNewLines, true, false, ";"));
  }
  if (this->outputType == "-vv") {
    emit stringResultReady(serverConnection->outputAsVV(query, saveToFile, showNewLines, true, ";"));
  }
  if (this->outputType == "-vvv") {
    emit stringResultReady(serverConnection->outputAsTable(query, true, saveToFile, showNewLines, true, true, ";"));
  }
  if (this->outputType == "-HTML") {
    emit stringResultReady(serverConnection->outputAsHTML(query, saveToFile, showNewLines, true, ";"));
  }
  if (this->outputType == "-XML") {
    emit stringResultReady(serverConnection->outputAsXML(query, saveToFile, showNewLines, true, ";"));
  }
  if (this->outputType == "-G") {
    emit stringResultReady(serverConnection->outputAsG(query, saveToFile, showNewLines, true, true, ";"));
  }
  if (this->outputType == "QList<QStringList>") {
    emit listResultReady(serverConnection->runQuery(query, false, false));
  }
  serverConnection->close();
  return;
}

ThreadedQueryExecution::ThreadedQueryExecution(QMap<QString, QVariant> serverConnectionParameters, QString query)
{
  this->serverConnectionParameters = serverConnectionParameters;
  this->query = query;
}

ThreadedQueryExecution::~ThreadedQueryExecution()
{

}

bool ThreadedQueryExecution::getOnlyExecuteSafeStatements()
{
  return onlyExecuteSafeStatements;
}

void ThreadedQueryExecution::setOnlyExecuteSafeStatements(bool onlyExecuteSafeStatements)
{
  this->onlyExecuteSafeStatements = onlyExecuteSafeStatements;
}

QString ThreadedQueryExecution::getOutputType()
{
  return this->outputType;
}

void ThreadedQueryExecution::setOutputType(QString outputType)
{
  this->outputType = outputType;
}

bool ThreadedQueryExecution::getSaveToFile()
{
  return this->saveToFile;
}

void ThreadedQueryExecution::setSaveToFile(bool saveToFile)
{
  this->saveToFile = saveToFile;
}

bool ThreadedQueryExecution::getShowNewLines()
{
  return this->showNewLines;
}

void ThreadedQueryExecution::setShowNewLines(bool showNewLines)
{
  this->showNewLines = showNewLines;
}

QString ThreadedQueryExecution::getQuery()
{
  return this->query;
}

void ThreadedQueryExecution::setQuery(QString query)
{
  this->query = query;
}

/*******************************************************************************************/

Trigger::Trigger(DBMS *serverConnection, QString triggerName, QString database)
{
  this->serverConnection = serverConnection;
  this->triggerName = StaticFunctions::quoteSymbol(triggerName);
  if (database.isEmpty())
    this->database = StaticFunctions::quoteSymbol(serverConnection->getDatabase());
  else
    this->database = StaticFunctions::quoteSymbol(database);
}

bool Trigger::drop()
{
  return serverConnection->executeQuery("DROP VIEW IF EXISTS " + formalName());
}

QString Trigger::formalName()
{
  return database + "." + triggerName;
}

QString Trigger::getDefinition()
{
  return serverConnection->runQuery("SHOW CREATE TRIGGER " + formalName())->at(0).at(2);
}

/*******************************************************************************************/

User::User(DBMS *serverConnection, QString userName)
{
  this->serverConnection = serverConnection;
  userName.isEmpty() ? this->userName = serverConnection->getFullUserName() : this->userName = userName;
}

bool User::drop()
{
  return serverConnection->executeQuery("DROP USER IF EXISTS " + this->userName);
}

QString User::getDefinition()
{
  return this->serverConnection->runQuery("SHOW CREATE USER " + this->userName)->at(0).at(0);
}

QStringList User::getGrants()
{
  return serverConnection->runQuerySingleColumn("SHOW GRANTS FOR " + this->userName);
}

bool User::hasGrant(QString grant, QString databaseObject)
{
  //Got not luck with QRegularExpression
//  QString userName = this->userName;
//  static QRegularExpression expression(".*" + grant + ".* ON "
//                                       + databaseObject.replace(".", "\\.").replace("*", "\\*").replace("`", "\\`")
//                                       + " TO "
//                                       + userName.replace("`", "\\`")
//                                       , QRegularExpression::CaseInsensitiveOption);
//  QRegularExpressionMatch regularExpressionMatch;
//  qDebug() << databaseObject << grant;
//  foreach (QString grantItem, getGrants()) {
//    regularExpressionMatch = expression.match(grantItem);
//    if (regularExpressionMatch.hasMatch()) {
//      qDebug() << databaseObject << grant << regularExpressionMatch.capturedTexts();
//      return true;
//    }
//  }
//  return false;
//  qDebug() << databaseObject << grant;

  foreach (QString grantItem, getGrants()) {
    if (grantItem.contains(grant.toUpper(), Qt::CaseInsensitive)
        &&
        grantItem.contains(" ON " + databaseObject + " TO " + this->userName)) {
//      qDebug() << databaseObject << grant << grantItem;
      return true;
    }
  }
  return false;
}

bool User::isRoot()
{
  bool hasRootGrants = false;
  foreach (QString grant, getGrants()) {
    if (grant.startsWith("GRANT ALL PRIVILEGES ON *.* TO " + this->userName)) {
      hasRootGrants = true;
      break;
    }
  }
  return hasRootGrants;
}

bool User::isRootAndGrantor()
{
  bool hasRootPrivilegesAndIsGrantor = false;
  foreach (QString grant, getGrants()) {
    if (grant.startsWith("GRANT ALL PRIVILEGES ON *.* TO " + this->userName)
        && grant.endsWith("WITH GRANT OPTION")) {
      hasRootPrivilegesAndIsGrantor = true;
      break;
    }
  }
  return hasRootPrivilegesAndIsGrantor;
}

QStringList User::getUserList()
{
  return serverConnection->runQuerySingleColumn("SELECT DISTINCT `User` FROM `mysql`.`user`");
}

QStringList User::getUserHostList(QString user)
{
  if (user.isEmpty())
    user = this->serverConnection->getUserName();
  return serverConnection->runQuerySingleColumn("SELECT `Host` FROM `mysql`.`user` WHERE `User` = '" + user + "'");
}
