/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef DatabaseComparision_H
#define DatabaseComparision_H

class QTreeWidgetItem;
class QTreeWidget;
class BaseTextEditor;
class DBMS;
class QPushButton;
class DTitleLabel;
class QGroupBox;

#include "dmdisubwindow.h"

/**
 * @brief The DatabaseComparision class allows for structural comparison between two databases.
 *
 * This window displays two lists of tables (primary and secondary connections) and
 * provides functionality to generate synchronization SQL scripts (diffs) to make
 * one database structure match the other.
 */
class DatabaseComparision : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the DatabaseComparision window.
   * @param serverConnection Pointer to the primary database connection.
   */
  DatabaseComparision(DBMS *serverConnection);

private:
  DBMS *serverConnection;
  DBMS *secondaryServerConnection;
  QList<QTreeWidgetItem *> tables;
  QList<QTreeWidgetItem *> secondaryTables;
  QTreeWidget *tablesListWidget;
  QTreeWidget *secondaryTablesListWidget;
  BaseTextEditor *resultEditor;
  QPushButton *pushButtonPrimaryCompare;
  QPushButton *pushButtonSecondaryCompare;

  /**
   * @brief Internal helper method to execute the comparison logic.
   * @param primary If true, compares Primary against Secondary. If false, compares Secondary against Primary.
   */
  void comparision(bool primary = true);

  DTitleLabel *dTitleLabel;
  QGroupBox *buttonGroup;

  /**
   * @brief Retranslates the user interface text.
   */
  void retranslateUI();

private slots:
  /**
   * @brief Populates the tree widgets with tables from both connections.
   */
  void fillTablesSlot();

  /**
   * @brief Slot triggered to compare the Primary DB against the Secondary DB.
   */
  void comparePrimarySlot();

  /**
   * @brief Slot triggered to compare the Secondary DB against the Primary DB.
   */
  void compareSeconadarySlot();

  /**
   * @brief Handles item activation (clicks) on the primary table list.
   * @param item The item clicked.
   * @param column The column index clicked.
   */
  void itemActivatedSlot(QTreeWidgetItem *item, int column);

  /**
   * @brief Handles item activation (clicks) on the secondary table list.
   * @param item The item clicked.
   * @param column The column index clicked.
   */
  void secondaryItemActivatedSlot(QTreeWidgetItem *item, int column);

  /**
   * @brief Displays a message on the status bar.
   * @param message The text to display.
   */
  void statusBarMessageSlot(QString message);

signals:
  /**
   * @brief Signal to request a status bar message in the main window.
   * @param message The text to display.
   */
  void statusBarMessage(QString message);

  /**
   * @brief Signal to report progress and status information.
   * @param message The status text.
   * @param timeout Duration to show the message.
   * @param progress The progress percentage (0-100).
   */
  void statusBarProgressMessage(const QString &message, const unsigned int timeout = 0, const double progress = 0);
};

#endif // DatabaseComparision_H
