/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef CONNECTDIALOG_H
#define CONNECTDIALOG_H

#include <QDialog>

#include "dbms.h"
#include "dsettings.h"

class QDialogButtonBox;
class QLineEdit;
class QSpinBox;
class QComboBox;
class QCheckBox;
class QMenu;
class QSignalMapper;
class QCompleter;
class DFileSelector;
class QAbstractButton;

/**
 * @brief The ConnectDialog class handles the database connection configuration.
 *
 * This dialog allows users to input connection details (host, port, user, password),
 * select database drivers, configure SSL settings, and manage saved connection profiles.
 */
class ConnectDialog : public QDialog
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the ConnectDialog.
   * @param serverConnection Pointer to the DBMS instance used to test or establish the connection.
   */
  ConnectDialog(DBMS *serverConnection);

  /**
   * @brief Retrieves the connection parameters entered by the user.
   * @return A QMap containing the configuration keys (e.g., Host, Port, User) and their values.
   */
  QMap<QString, QVariant> getValues();

  /**
   * @brief Checks if a successful connection was established during the dialog's lifecycle.
   * @return True if connected, false otherwise.
   */
  bool getConnectionPerformed();

private slots:
  /**
   * @brief Validates the input fields.
   *
   * Enables or disables the "Connect" button based on whether required fields
   * (like host and username) are filled.
   */
  void validateInputs();

  /**
   * @brief Populates the input fields based on a selected connection profile.
   * @param text The name of the selected connection profile.
   */
  void fillLineEdits(const QString &text);

  /**
   * @brief Handles changes to the connection type (Database Driver).
   * @param text The name of the selected driver (e.g., "QMYSQL", "QPSQL").
   */
  void comboConnectionTypeSlot(const QString &text);

  /**
   * @brief Prepares and displays the available databases menu.
   */
  void databasesMenuSlot();

  /**
   * @brief Updates the database field when a specific database is selected from the menu.
   * @param database The name of the selected database.
   */
  void changeDatabaseSlot(QString database);

  /**
   * @brief Prepares and displays the available collations menu.
   */
  void collatoinsMenuSlot();

  /**
   * @brief Updates the collation field when a specific collation is selected.
   * @param collation The name of the selected collation.
   */
  void changeCollationSlot(QString collation);

  /**
   * @brief Toggles the visibility of SSL (Secure Socket Layer) configuration fields.
   * @param state The state of the checkbox (Qt::Checked or Qt::Unchecked).
   */
  void useASSLConnectionStateChanged(int state);

  /**
   * @brief Toggles the visibility of the password characters.
   * @param checked True to show the password in plain text, false to mask it.
   */
  void maskPasswordToggled(bool checked);

  /**
   * @brief Handles clicks on the standard dialog buttons (Ok/Cancel).
   * @param button The button that was clicked.
   */
  void actionRoleSlot(QAbstractButton *button);

private:
  QDialogButtonBox *buttonBox;
  QLineEdit *lineEditServer;
  QSpinBox *spinBoxPort;
  QLineEdit *lineEditUser;
  QLineEdit *lineEditPassword;
  QLineEdit *lineEditDatabase;
  QComboBox *comboConnectionName;
  QCheckBox *storePasswords;
  DSettings settings;
  QComboBox *comboConnectionType;
  QPushButton *databasesPushButton;
  QMenu *databasesMenu;
  QSignalMapper *databasesMapper;
  bool connectionPerformed;
  DBMS *serverConnection;
  QCompleter *completer;

  /**
   * @brief Internal helper to configure DBMS settings.
   */
  void setDBMS();
  unsigned int count;
  QCheckBox *sortConnectionList;
  QMenu *collationsMenu;
  QLineEdit *lineEditCollation;
  QPushButton *collationPushButton;
  QSignalMapper *collationsMapper;
  QCheckBox *useASSLConnection;
  DFileSelector *fileSelectorClientKey;
  DFileSelector *fileSelectorClientCert;
  DFileSelector *fileSelectorCA;
  QPushButton *maskPassword;
};

#endif // CONNECTDIALOG_H
