/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QLabel>
#include <QLineEdit>
#include <QSpinBox>
#include <QFormLayout>
#include <QGroupBox>
#include <QDialogButtonBox>
#include <QPushButton>
#include <QComboBox>
#include <QCheckBox>
#include <QApplication>
#include <QMenu>
#include <QSignalMapper>
#include <QCompleter>
#include <QMessageBox>

#include "connectdialog.h"
#include "dtitlelabel.h"
#include "staticfunctions.h"
#include "dfileselector.h"
#include "dicon.h"

#include "QDebug"

ConnectDialog::ConnectDialog(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
  count = 0;
  setWindowTitle(tr("Connect to a Database Server"));
  setObjectName(windowTitle());
  setWindowIcon(DIcon::Caliope());

  comboConnectionName = new QComboBox();
  comboConnectionName->setAttribute(Qt::WA_AlwaysShowToolTips, true);
  comboConnectionName->setToolTip(tr("List of aviable database connections"));
  comboConnectionName->setStatusTip(comboConnectionName->toolTip());
  comboConnectionName->setWhatsThis(comboConnectionName->toolTip());
  comboConnectionName->setEditable(true);
  QStringList connections = settings.allKeys("ServerConnections");
  int valueToRemove = connections.indexOf("StorePassword");
  if (valueToRemove >= 0)
      connections.removeAt(valueToRemove);
  valueToRemove = connections.indexOf("SortConnectionList");
  if (valueToRemove >= 0)
      connections.removeAt(valueToRemove);

  sortConnectionList = new QCheckBox(tr("Sort connection list by used times"));
  sortConnectionList->setCheckState(settings.value("ServerConnections/SortConnectionList", false).toBool() ? Qt::Checked : Qt::Unchecked);

  if (sortConnectionList->isChecked()) {
    QMultiMap<int, QString> connectionsMap;
    foreach (QString connection, connections)
      connectionsMap.insert(settings.value("ServerConnections/" + connection).toMap().value("ConnectionsCount").toInt() * -1, connection);
    comboConnectionName->insertItems(0, connectionsMap.values());
    connect(comboConnectionName, SIGNAL(textActivated(QString)), this, SLOT(fillLineEdits(QString)));
  } else {
    comboConnectionName->insertItems(0, connections);
  }

  completer = new QCompleter(connections, comboConnectionName);
  completer->setFilterMode(Qt::MatchContains);
  completer->setCaseSensitivity(Qt::CaseInsensitive);
  comboConnectionName->setCompleter(completer);

  comboConnectionType = new QComboBox();
  comboConnectionType->insertItems(0, StaticFunctions::dbmsEnabled());
  connect(comboConnectionType, SIGNAL(textActivated(QString)), this, SLOT(comboConnectionTypeSlot(QString)));

  lineEditServer = new QLineEdit;
  lineEditServer->setPlaceholderText(tr("IP address or server name"));
  lineEditServer->setClearButtonEnabled(true);
  connect(lineEditServer, SIGNAL(textChanged(QString)), this, SLOT(validateInputs()));

  spinBoxPort = new QSpinBox;
  spinBoxPort->setRange(0, 65535);

  lineEditUser = new QLineEdit;
  lineEditUser->setPlaceholderText(tr("Username"));
  lineEditUser->setClearButtonEnabled(true);
  connect(lineEditUser, SIGNAL(textChanged(QString)), this, SLOT(validateInputs()));

  lineEditPassword = new QLineEdit;
  lineEditPassword->setPlaceholderText(tr("Password"));
  lineEditPassword->setEchoMode(QLineEdit::Password);
  lineEditPassword->setClearButtonEnabled(true);
  connect(lineEditPassword, SIGNAL(textChanged(QString)), this, SLOT(validateInputs()));
  QHBoxLayout *passwordHlLayout = new QHBoxLayout;
  passwordHlLayout->addWidget(lineEditPassword);
  maskPassword = new QPushButton(DIcon::ObjectLocked(), "", this);
  maskPassword->setCheckable(true);
  connect(maskPassword, SIGNAL(toggled(bool)), this, SLOT(maskPasswordToggled(bool)));
  passwordHlLayout->addWidget(maskPassword);

  databasesMenu = new QMenu(this);
  databasesMenu->setIcon(DIcon::Database());
  connect(databasesMenu, SIGNAL(aboutToShow()), this, SLOT(databasesMenuSlot()));
  lineEditDatabase = new QLineEdit;
  lineEditDatabase->setPlaceholderText(tr("Database"));
  lineEditDatabase->setClearButtonEnabled(true);
  databasesPushButton = new QPushButton(DIcon::Database(), "", this);
  databasesPushButton->setMenu(databasesMenu);
  QHBoxLayout *databaseHlLayout = new QHBoxLayout;
  databaseHlLayout->addWidget(lineEditDatabase);
  databaseHlLayout->addWidget(databasesPushButton);
  databasesMapper = new QSignalMapper(this);
  connect(databasesMapper, SIGNAL(mappedString(QString)), this, SLOT(changeDatabaseSlot(QString)));
  connectionPerformed = false;

  storePasswords = new QCheckBox(tr("Store passwords"));
  storePasswords->setCheckState(settings.value("ServerConnections/StorePassword", false).toBool() ? Qt::Checked : Qt::Unchecked);

  collationsMenu = new QMenu(this);
  collationsMenu->setIcon(DIcon::CharacterSet());
  connect(collationsMenu, SIGNAL(aboutToShow()), this, SLOT(collatoinsMenuSlot()));
  lineEditCollation = new QLineEdit;
  lineEditCollation->setPlaceholderText(tr("Collation"));
  lineEditCollation->setClearButtonEnabled(true);
  collationPushButton = new QPushButton(DIcon::CharacterSet(), "", this);
  collationPushButton->setMenu(collationsMenu);
  QHBoxLayout *collationHlLayout = new QHBoxLayout;
  collationHlLayout->addWidget(lineEditCollation);
  collationHlLayout->addWidget(collationPushButton);

  collationsMapper = new QSignalMapper(this);
  connect(collationsMapper, SIGNAL(mappedString(QString)), this, SLOT(changeCollationSlot(QString)));

  useASSLConnection = new QCheckBox(tr("Use a SSL connection"));
  useASSLConnection->setCheckState(settings.value("ServerConnections/SSLConnection", false).toBool() ? Qt::Checked : Qt::Unchecked);
  connect(useASSLConnection, SIGNAL(stateChanged(int)), this, SLOT(useASSLConnectionStateChanged(int)));

  fileSelectorClientKey = new DFileSelector(DFileSelectorContexts::PEMKey, tr("SSL Client Key file:"), false, DIcon::PEMKey());
  fileSelectorClientCert = new DFileSelector(DFileSelectorContexts::PEMCert, tr("SSL Client Cert file:"), false, DIcon::PEMCert());
  useASSLConnectionStateChanged(settings.value("ServerConnections/SSLConnection", 0).toInt());

  QFormLayout *formLayout = new QFormLayout;
  formLayout->addRow(tr("&Connection Name:"), comboConnectionName);
  formLayout->addRow(sortConnectionList);
  formLayout->addRow(tr("Connection &Type:"), comboConnectionType);
  formLayout->addRow(tr("&Server:"), lineEditServer);
  formLayout->addRow(tr("&Port:"), spinBoxPort);
  formLayout->addRow(tr("&User:"), lineEditUser);
  formLayout->addRow(tr("Password:"), passwordHlLayout);
  formLayout->addRow(tr("Database:"), databaseHlLayout);
  formLayout->addRow(tr("Collation:"), collationHlLayout);
  formLayout->addRow(storePasswords);
  formLayout->addRow(useASSLConnection);
  formLayout->addRow(fileSelectorClientKey);
  formLayout->addRow(fileSelectorClientCert);
  QGroupBox *mainGroupBox = new QGroupBox(windowTitle());
  mainGroupBox->setLayout(formLayout);

  buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel);
  buttonBox->addButton(tr("Ping"), QDialogButtonBox::ActionRole);
  buttonBox->addButton(tr("New connection"), QDialogButtonBox::ActionRole);
  connect(buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
  connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
  connect(buttonBox, SIGNAL(clicked(QAbstractButton*)), this, SLOT(actionRoleSlot(QAbstractButton*)));

  QVBoxLayout *verticalLayout = new QVBoxLayout;
  verticalLayout->addWidget(new DTitleLabel(windowTitle()));
  verticalLayout->addWidget(mainGroupBox);
  verticalLayout->addWidget(buttonBox);
  setLayout(verticalLayout);

  fillLineEdits(comboConnectionName->currentText());
  comboConnectionName->setFocus(Qt::OtherFocusReason);
  comboConnectionName->lineEdit()->selectAll();
}

void ConnectDialog::fillLineEdits(const QString &text)
{
  QMap<QString, QVariant> params = settings.value("ServerConnections/" + text).toMap();
  comboConnectionType->setCurrentIndex(comboConnectionType->findText(params.value("ConnectionType").toString()));
  lineEditUser->setText(params.value("User").toString());
  lineEditUser->setToolTip(lineEditUser->text());
  lineEditServer->setText(params.value("Host").toString());
  lineEditServer->setToolTip(lineEditServer->text());
  spinBoxPort->setValue(params.value("Port").toInt());
  lineEditDatabase->setText(params.value("Database").toString());
  lineEditDatabase->setToolTip(lineEditDatabase->text());
  if (storePasswords->isChecked())
    lineEditPassword->setText(StaticFunctions::password(params.value("Password").toString()));
  setDBMS();
  count = params.value("ConnectionsCount").toInt();
  lineEditCollation->setText(params.value("Collation").toString());
  lineEditCollation->setToolTip(lineEditCollation->text());
  useASSLConnection->setCheckState(params.value("UseSSL") == "1" ? Qt::Checked : Qt::Unchecked);
  fileSelectorClientKey->setFileName(params.value("KeyFile").toString());
  fileSelectorClientCert->setFileName(params.value("CertFile").toString());
  buttonBox->button(QDialogButtonBox::Ok)->setFocus(Qt::TabFocusReason);
}

void ConnectDialog::comboConnectionTypeSlot(const QString &text)
{
  setDBMS();
  if (text == "--")
    spinBoxPort->setValue(0);
  if (text == "MySQL" || text == "MariaDB")
    spinBoxPort->setValue(3306);
}

void ConnectDialog::databasesMenuSlot()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  databasesMenu->clear();
  if (!serverConnection->isOpened()) {
    serverConnection->setUserName(lineEditUser->text());
    serverConnection->setHostName(lineEditServer->text());
    serverConnection->setPassword(lineEditPassword->text());
    serverConnection->setDatabase(StaticFunctions::DBMSDefaultDatabase());
    serverConnection->setPort(spinBoxPort->value());
    serverConnection->setCharacterSet(lineEditCollation->text().split("|").at(0));
    serverConnection->setCollation(lineEditCollation->text().split("|").at(1));
    serverConnection->setUseSSL(useASSLConnection->isChecked());
    serverConnection->setKeyFile(fileSelectorClientKey->getFileName());
    serverConnection->setCertFile(fileSelectorClientCert->getFileName());
    serverConnection->setConnectionName(comboConnectionName->currentText());
    serverConnection->open();
    connectionPerformed = true;
  }
  foreach (QString database, serverConnection->getDatabases()) {
    QAction *action = databasesMenu->addAction(database);
    action->setCheckable(true);
    action->setChecked(action->text() == lineEditDatabase->text());
    connect(action, SIGNAL(triggered()), databasesMapper, SLOT(map()));
    databasesMapper->setMapping(action, database);
  }
  QApplication::restoreOverrideCursor();
}

void ConnectDialog::changeDatabaseSlot(QString database)
{
  lineEditDatabase->setText(database);
  lineEditDatabase->setToolTip(lineEditDatabase->text());
  serverConnection->changeDatabase(database);
  getValues();
}

void ConnectDialog::collatoinsMenuSlot()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  collationsMenu->clear();
  if (!serverConnection->isOpened()) {
    serverConnection->setUserName(lineEditUser->text());
    serverConnection->setHostName(lineEditServer->text());
    serverConnection->setPassword(lineEditPassword->text());
    serverConnection->setDatabase(StaticFunctions::DBMSDefaultDatabase());
    serverConnection->setPort(spinBoxPort->value());
    serverConnection->setCharacterSet(lineEditCollation->text().split("|").at(0));
    serverConnection->setCollation(lineEditCollation->text().split("|").at(1));
    serverConnection->setUseSSL(useASSLConnection->isChecked());
    serverConnection->setKeyFile(fileSelectorClientKey->getFileName());
    serverConnection->setCertFile(fileSelectorClientCert->getFileName());
    serverConnection->setConnectionName(comboConnectionName->currentText());
    serverConnection->open();
    connectionPerformed = true;
  }
  QList<QStringList> *collations = serverConnection->getCollationsApplicability();
  for (int row = 0; row < collations->count() - 1; row++) {
    QAction *action = collationsMenu->addAction(collations->at(row).at(1) + "|" + collations->at(row).at(0));
    action->setCheckable(true);
    action->setChecked(action->text() == lineEditCollation->text());
    connect(action, SIGNAL(triggered()), collationsMapper, SLOT(map()));
    collationsMapper->setMapping(action, collations->at(row).at(1) + "|" + collations->at(row).at(0));
  }
  QApplication::restoreOverrideCursor();
}

void ConnectDialog::changeCollationSlot(QString collation)
{
  lineEditCollation->setText(collation);
  lineEditCollation->setToolTip(lineEditCollation->text());
  serverConnection->setCharsetAndCollation(collation.split("|").at(0), collation.split("|").at(1));
  getValues();
}

void ConnectDialog::useASSLConnectionStateChanged(int state)
{
  fileSelectorClientKey->setEnabled(state);
  fileSelectorClientCert->setEnabled(state);
}

void ConnectDialog::maskPasswordToggled(bool checked)
{
  lineEditPassword->setEchoMode(checked ? QLineEdit::Normal : QLineEdit::Password);
  maskPassword->setIcon(checked ? DIcon::ObjectUnlocked() : DIcon::ObjectLocked());
}

void ConnectDialog::actionRoleSlot(QAbstractButton *button)
{
  if (button->text() == tr("Ping")) {
    if (!serverConnection->isOpened()) {
      serverConnection->setUserName(lineEditUser->text());
      serverConnection->setHostName(lineEditServer->text());
      serverConnection->setPassword(lineEditPassword->text());
      serverConnection->setDatabase(StaticFunctions::DBMSDefaultDatabase());
      serverConnection->setPort(spinBoxPort->value());
      serverConnection->setCharacterSet(lineEditCollation->text().split("|").at(0));
      serverConnection->setCollation(lineEditCollation->text().split("|").at(1));
      serverConnection->setUseSSL(useASSLConnection->isChecked());
      serverConnection->setKeyFile(fileSelectorClientKey->getFileName());
      serverConnection->setCertFile(fileSelectorClientCert->getFileName());
      serverConnection->setConnectionName(comboConnectionName->currentText());
      serverConnection->open();
      connectionPerformed = true;
    }
    if (serverConnection->ping() == 0)
      QMessageBox::information(this, tr("Ping successful to: %1").arg(serverConnection->getHostName()), tr("Ping successful"));
  } else if (button->text() == tr("New connection")) {
    comboConnectionName->setCurrentText(tr("New connection"));
    comboConnectionType->setCurrentText("MariaDB");
    lineEditUser->setText("root");
    lineEditServer->setText("localhost");
    spinBoxPort->setValue(3306);
    lineEditDatabase->setText("mysql");
    lineEditCollation->setText("utf8mb4|utf8mb4_general_ci");
    useASSLConnection->setCheckState(Qt::Unchecked);
    fileSelectorClientKey->setFileName("");
    fileSelectorClientCert->setFileName("");
    lineEditPassword->clear();
  }
}

void ConnectDialog::setDBMS()
{
  if (comboConnectionType->currentText() == "--")
    serverConnection->setDBMSType(StaticFunctions::Undefined);
  if (comboConnectionType->currentText() == "MySQL")
    serverConnection->setDBMSType(StaticFunctions::MySQL);
  if (comboConnectionType->currentText() == "MariaDB")
    serverConnection->setDBMSType(StaticFunctions::MariaDB);
}

QMap<QString, QVariant> ConnectDialog::getValues()
{
  QMap<QString, QVariant> connectionParameters;
  connectionParameters.insert("ConnectionType", comboConnectionType->currentText());
  connectionParameters.insert("Name", comboConnectionName->currentText());
  connectionParameters.insert("User", lineEditUser->text());
  connectionParameters.insert("Host", lineEditServer->text());
  connectionParameters.insert("Port", QString("%1").arg(spinBoxPort->value()));
  connectionParameters.insert("Database", lineEditDatabase->text());
  connectionParameters.insert("ConnectionsCount", QString("%1").arg(++count));
  connectionParameters.insert("Collation", lineEditCollation->text());
  connectionParameters.insert("UseSSL", QString("%1").arg(useASSLConnection->isChecked()));
  connectionParameters.insert("KeyFile", fileSelectorClientKey->getFileName().isEmpty() ? "-" : fileSelectorClientKey->getFileName());
  connectionParameters.insert("CertFile", fileSelectorClientCert->getFileName().isEmpty() ? "-" : fileSelectorClientCert->getFileName());
  connectionParameters.insert("Password", storePasswords->isChecked() ? StaticFunctions::password(lineEditPassword->text(), true) : "");

  //Using the ServerConnections
  settings.setValue("ServerConnections/" + comboConnectionName->currentText(), connectionParameters);
  settings.setValue("ServerConnections/StorePassword", storePasswords->isChecked());
  settings.setValue("ServerConnections/SortConnectionList", sortConnectionList->isChecked());
  return connectionParameters;
}

bool ConnectDialog::getConnectionPerformed()
{
  if (connectionPerformed)
    serverConnection->setDatabase(lineEditDatabase->text());
  return connectionPerformed;
}

void ConnectDialog::validateInputs()
{
  buttonBox->button(QDialogButtonBox::Ok)->setEnabled((!lineEditServer->text().isEmpty() && !lineEditUser->text().isEmpty()));
}
