/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef CODESNIPPETS_H
#define CODESNIPPETS_H

#include <QWidget>

#include "dsettings.h"

class QPlainTextEdit;
class QDialogButtonBox;

/**
 * @brief The CodeSnippets class manages reusable code fragments.
 *
 * This widget allows the user to view, edit, save, and retrieve code snippets.
 * It utilizes DSettings for persistent storage of these snippets.
 */
class CodeSnippets : public QWidget
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the CodeSnippets widget.
   *
   * Initializes the user interface components and loads existing snippets from settings.
   */
  CodeSnippets();

  /**
   * @brief Retranslates the user interface.
   *
   * Updates the text of buttons and labels to reflect the current language settings.
   */
  void retranslateUI();

  /**
   * @brief Retrieves a list of available code snippet identifiers or titles.
   * @return A QStringList containing the keys or names of stored snippets.
   */
  QStringList getCodeSnippets();

  /**
   * @brief Retrieves the content of a specific code snippet.
   * @param key The identifier or title of the snippet to retrieve.
   * @return A QString containing the code content of the requested snippet.
   */
  QString getCodeSnippet(const QString key);

  /**
   * @brief Adds a new code snippet to the collection.
   * @param snippet The actual code content.
   * @param title The title or identifier for the snippet.
   */
  void addSnippet(QString snippet, QString title);

private:
  DSettings settings;
  QPlainTextEdit *codeSnippetsPlainTextEdit;
  QDialogButtonBox *buttonBox;

  /**
   * @brief Loads code snippets from persistent storage.
   * @return A QString containing the raw loaded data (formatted for the text editor).
   */
  QString loadCodeSnippets();

private slots:
  /**
   * @brief Displays help information regarding code snippets.
   */
  void helpCodeSnippets();

  /**
   * @brief Saves the current state of the code snippets to persistent storage.
   */
  void saveCodeSnippets();
};

#endif // CODESNIPPETS_H
