/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef CATALOGS_H
#define CATALOGS_H

#include <QDialog>
#include <QModelIndex>
#include <QStandardItem>
#include <QSystemTrayIcon>

class QTabWidget;
class QToolBar;
class QAction;
class QComboBox;
class QMenu;
class NewDatabase;
class BaseTextEditor;
class DTableView;
class DTitleLabel;
class TextEditor;
class Projects;

#include "dmdisubwindow.h"
#include "dbms.h"

/**
 * @brief Container class for database object types used in the editor.
 */
class ItemEditorTypes
{
public:
  /**
   * @brief Enumeration representing the specific type of database object being edited.
   */
  enum ItemEditorType {Table, View, Index, Field, Routine, Trigger, Event};
};

/**
 * @brief A dialog window for creating or editing specific database objects (DDL).
 *
 * This class provides a text editor interface for users to write and apply
 * SQL statements to create or modify tables, views, triggers, etc.
 */
class ItemEditor : public QDialog
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the ItemEditor dialog.
   * @param serverConnection Pointer to the active DBMS connection.
   * @param title The title of the dialog window.
   * @param type The type of object being edited (e.g., Table, View).
   */
  ItemEditor(DBMS *serverConnection, QString title, ItemEditorTypes::ItemEditorType type);

  /**
   * @brief Pointer to the text editor widget containing the SQL statement.
   */
  TextEditor *editor;

private:
  ItemEditorTypes::ItemEditorType itemType;

signals:
  /**
   * @brief Emitted when a URL link is activated within the editor (e.g., help links).
   * @param url The string representation of the URL.
   */
  void openURL(QString url);

private slots:
  /**
   * @brief Executes the SQL statement currently in the editor.
   */
  void applyStatement();

  /**
   * @brief Displays help information relevant to the current object type.
   */
  void help();
};

/**
 * @brief The Catalogs class acts as the main browser for database schemas.
 *
 * Inheriting from DMdiSubWindow, this class displays tabs containing lists of
 * tables, views, indexes, fields, routines, triggers, and events. It facilitates
 * management operations such as creating, deleting, and inspecting these objects.
 */
class Catalogs : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the Catalogs window.
   * @param serverConnection Pointer to the active DBMS connection.
   */
  Catalogs(DBMS *serverConnection);

  /**
   * @brief Flag indicating if the catalog is in read-only mode.
   */
  bool readOnly;

  /**
   * @brief Retranslates the user interface text for language changes.
   */
  void retranslateUI();

public slots:
  /**
   * @brief Refreshes the information of the currently selected object type.
   */
  void updateObjectInfo();

  /**
   * @brief Triggers the interface for creating a new database.
   */
  void newDatabaseActionTriggered();

  /**
   * @brief Triggers the logic for deleting the currently selected database.
   */
  void deleteDatabaseActionTriggered();

  /**
   * @brief Toggles the read-only state of the catalog viewer.
   * @param readOnly True to enable read-only mode, false to disable.
   */
  void setReadOnly(bool readOnly);

  /**
   * @brief Triggers the action to populate a table with dummy data.
   */
  void populateTableActionTriggered();

private:
  DBMS *serverConnection;
  QTabWidget *catalogsTab;
  DTableView *tablesDTableView;
  DTableView *viewsDTableView;
  DTableView *indexesDTableView;
  DTableView *fieldsDTableView;
  DTableView *routinesDTableView;
  DTableView *triggersDTableView;
  DTableView *eventsDTableView;
  BaseTextEditor *readOnlyEditor;
  QList<QStringList> *result;
  ItemEditor *itemEditor;
  NewDatabase *newDatabase;
  int lastRow;

  /**
   * @brief Retrieves and displays index information for a specific table.
   * @param table The name of the table.
   */
  void indexesInfo(QString table);

  /**
   * @brief Retrieves and displays field (column) information for a specific table.
   * @param table The name of the table.
   */
  void fieldsInfo(QString table);

  /**
   * @brief Retrieves and displays trigger information for a specific table.
   * @param table The name of the table.
   */
  void triggersInfo(QString table);

  /**
   * @brief Logic to delete selected tables.
   */
  void deleteTables();

  /**
   * @brief Logic to delete selected views.
   */
  void deleteViews();

  /**
   * @brief Logic to delete selected indexes.
   */
  void deleteIndexes();

  /**
   * @brief Logic to delete selected fields (columns).
   */
  void deleteFields();

  /**
   * @brief Logic to delete selected routines (procedures/functions).
   */
  void deleteRoutines();

  /**
   * @brief Updates the window title based on the current context.
   */
  void setTitle();

  DTitleLabel *dTitleLabel;

  /**
   * @brief Generates a dummy value for table population based on data type.
   * @param dataType The SQL data type (e.g., INT, VARCHAR).
   * @param length The length constraint of the field (default is 1).
   * @return A string representing the generated value.
   */
  QString valueToPopulate(QString dataType, unsigned int length = 1);

private slots:
  /**
   * @brief Slot to fetch and display the list of tables.
   */
  void tablesInfo();

  /**
   * @brief Slot to fetch and display the list of views.
   */
  void viewsInfo();

  /**
   * @brief Slot to fetch and display the list of stored routines.
   */
  void routinesInfo();

  /**
   * @brief Slot to fetch and display the list of events.
   */
  void eventsInfo();

  /**
   * @brief Handle clicks on the tables list.
   * @param modelIndex The index of the clicked item.
   */
  void tablesClicked(QModelIndex modelIndex);

  /**
   * @brief Handle clicks on the views list.
   * @param modelIndex The index of the clicked item.
   */
  void viewsClicked(QModelIndex modelIndex);

  /**
   * @brief Handle clicks on the routines list.
   * @param modelIndex The index of the clicked item.
   */
  void routinesClicked(QModelIndex modelIndex);

  /**
   * @brief Handle clicks on the triggers list.
   * @param modelIndex The index of the clicked item.
   */
  void triggersClicked(QModelIndex modelIndex);

  /**
   * @brief Handle clicks on the events list.
   * @param modelIndex The index of the clicked item.
   */
  void eventsClicked(QModelIndex modelIndex);

  /**
   * @brief Generic slot to handle the "New Object" action.
   */
  void newObjectSlot();

  /**
   * @brief Generic slot to handle the "Edit Object" action.
   */
  void editObjectSlot();

  /**
   * @brief Generic slot to handle the "Delete Object" action.
   */
  void deleteObjectSlot();

  /**
   * @brief Slot called when an item in the tables list is modified.
   * @param item The item that was changed.
   */
  void tablesItemChangedSlot(QStandardItem *item);

  /**
   * @brief Relays progress messages to the main application status bar.
   * @param message The message text.
   * @param timeout Duration to show the message (ms).
   * @param progress Progress percentage (0-100).
   */
  void statusBarProgressMessageSlot(const QString &message, const unsigned int timeout = 0, const double progress = 0);

signals:
  /**
   * @brief Signal emitted when a loading process begins.
   * @param message Description of the task.
   * @param timeout Expected timeout or duration.
   * @param progress Initial progress value.
   */
  void loadStarted(QString message, unsigned int timeout, double progress);

  /**
   * @brief Signal emitted when a loading process finishes.
   * @param message Completion message.
   * @param timeout Duration to show completion message.
   * @param progress Final progress value (usually 100).
   */
  void loadFinished(QString message, unsigned int timeout, double progress);

  /**
   * @brief Signal emitted to update loading progress.
   * @param message Current status message.
   * @param timeout Duration for the message.
   * @param progress Current progress value.
   */
  void loadProgress(QString message, unsigned int timeout, double progress);

  /**
   * @brief Signal to display a progress message on the status bar.
   * @param message The message text.
   * @param timeout Duration to show the message.
   * @param progress Progress percentage.
   */
  void statusBarProgressMessage(const QString &message, const unsigned int timeout = 0, const double progress = 0);

  /**
   * @brief Signal to show a temporary popup message on the status bar.
   * @param message The message text.
   * @param timeout Duration to show the message.
   */
  void statusBarMessagePopup(const QString &message, const int timeout = 0);

  /**
   * @brief Signal to show a general message on the status bar, optionally with an icon.
   * @param message The message text.
   * @param icon The icon type (e.g., Information, Warning).
   * @param timeout Duration to show the message.
   */
  void statusBarMessage(const QString &message, QSystemTrayIcon::MessageIcon icon = QSystemTrayIcon::Information, int timeout = 0);
};

#endif // CATALOGS_H
