/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef BASETEXTEDITOR_H
#define BASETEXTEDITOR_H

#include <QPlainTextEdit>

#include "editortypes.h"
#include "dsettings.h"

class GeneralHighlighter;
class FoldingTextHandler;
class FramingTextHandler;

/**
 * @brief The BaseTextEditor class provides the core text editing functionality.
 *
 * This class extends QPlainTextEdit to support advanced features required by Calíope,
 * including syntax highlighting, code folding, line numbering, bracket matching,
 * and smart indentation.
 */
class BaseTextEditor : public QPlainTextEdit
{
  Q_OBJECT
  // Q_INTERFACES(QTextObjectInterface)

public:
  /**
   * @brief Constructs a new BaseTextEditor.
   * @param type The type of editor to initialize (e.g., specific language support).
   */
  BaseTextEditor(EditorTypes::EditorType type);

  /**
   * @brief Handles the painting event for the line number area.
   * @param event The paint event parameters.
   */
  void lineNumberAreaPaintEvent(QPaintEvent *event);

  /**
   * @brief Calculates the required width for the line number area.
   * @return The width in pixels based on the number of digits in the line count.
   */
  int lineNumberAreaWidth();

  /** @brief Pointer to the widget responsible for drawing line numbers. */
  QWidget *lineNumberArea;

  /** @brief Flag indicating whether the line number area should be shown. */
  bool showNumberArea;

  /**
   * @brief Forces a re-highlighting of the text based on a specific expression.
   * @param exp The regular expression or rule to apply.
   */
  void rehighlight(const QString &exp);

  /** @brief Action to increase indentation of the selected text. */
  QAction *indentAction;

  /** @brief Action to decrease indentation of the selected text. */
  QAction *unindentAction;

  /** @brief List of bookmarks or specific points of interest in the document. */
  QStringList pointList;

  /**
   * @brief Moves the cursor to a specific line number.
   * @param lineNumber The target line number (1-based index).
   * @param center If true, centers the line in the viewport.
   */
  void gotoLine(unsigned int lineNumber = 0, bool center = true);

  /**
   * @brief Retranslates the UI elements.
   * Called when the application language changes dynamically.
   */
  void retranslateUI();

  /** @brief Action to insert a standard license header template. */
  QAction *insertLicenceTemplateAction;

  /** @brief List containing the indices of rows that can be folded. */
  QVector<unsigned int> *foldRowList;

  /** @brief Map storing the hierarchy of folded rows. */
  QMap<unsigned int, QVector<unsigned int>> *foldedRowList;

  /** @brief Action to toggle the visualization of white spaces and tabs. */
  QAction *visualizeSpacesAction;

  /**
   * @brief Retrieves a list of words currently in the document.
   * Used for autocompletion suggestions.
   * @return A list of unique words.
   */
  QStringList wordList();

  /** @brief Action to fold the current code block. */
  QAction *foldAction;

  /** @brief Action to unfold the current code block. */
  QAction *unfoldAction;

  /**
   * @brief Identifies matching brackets around the cursor.
   * @return A vector containing the positions of the matching brackets.
   */
  QVector<unsigned int> matchBracket();

  /** @brief Stores the absolute positions of the last successfully matched brackets. */
  QVector<unsigned int> absolutePostionForTheLastBracketMatched;

  /**
   * @brief Draws a custom rectangle on the editor's viewport.
   * @param rect The geometry of the rectangle to draw.
   */
  void drawRectangle(QRect rect);

  /** @brief List of rows that have been modified since load/save. */
  QVector<unsigned int> *modifiedRowList;

  /**
   * @brief Marks a specific row as modified.
   * @param item The index of the row to mark.
   */
  void appendModifiedRowList(unsigned int item);

  /** @brief Stores the file content line by line for internal processing. */
  QStringList fileLineByLine;

  /** @brief Flag indicating if the current line is part of a folded block. */
  bool lineFolded;

  /**
   * @brief Sets the font weight for the editor text.
   * @param weight The font weight (e.g., QFont::Bold, QFont::Normal).
   */
  void setFontWeight(QFont::Weight weight = QFont::Normal);

protected:
  /**
   * @brief Handles resize events to adjust the line number area.
   * @param event The resize event details.
   */
  void resizeEvent(QResizeEvent *event);

  /**
   * @brief Handles key press events for shortcuts and editing logic.
   * @param event The key event details.
   */
  void keyPressEvent(QKeyEvent *event);

  /**
   * @brief Handles mouse double-click events (e.g., word selection).
   * @param event The mouse event details.
   */
  void mouseDoubleClickEvent(QMouseEvent *event);

  // void mouseMoveEvent(QMouseEvent *event);

  /**
   * @brief Handles drag and drop events (e.g., opening a file).
   * @param e The drop event details.
   */
  void dropEvent(QDropEvent *e);

private slots:
  /** @brief Updates the width of the line number area when the block count changes. */
  void updateLineNumberAreaWidth();

  /**
   * @brief Updates the line number area content during scrolling.
   * @param rect The rectangle to update.
   * @param dy The vertical scrolling amount.
   */
  void updateLineNumberArea(const QRect &rect, int dy);

  /** @brief Slot triggered to perform indentation. */
  void indentActionTriggered();

  /** @brief Slot triggered to perform unindentation. */
  void unindentActionTriggered();

  /**
   * @brief Toggles the visibility of non-printable characters (spaces/tabs).
   * @param toggled True to show spaces, false to hide them.
   */
  void visualizeSpacesActionTriggered(bool toggled);

  /** @brief Slot triggered to fold the code block at the cursor. */
  void foldActionTriggered();

  /** @brief Slot triggered to unfold the code block at the cursor. */
  void unfoldActionTriggered();

  /** @brief Slot triggered whenever the text content changes. */
  void textChangedSlot();

public slots:
  /** @brief Highlights the background of the current line. */
  void highlightCurrentLine();

  /**
   * @brief Shows or hides the line number sidebar.
   * @param show True to show, false to hide.
   */
  void showHideLineNumbers(bool show);

  /**
   * @brief Toggles a bookmark/dockmark at the current line.
   * @param pos The position context (usually from mouse click).
   */
  void dockmarkCurrentLine(const QPoint &pos);

  /** @brief Inserts the license template at the current cursor position. */
  void insertLicenceTemplateSlot();

  /**
   * @brief Sets the content of the editor.
   * @param text The string content to set.
   * @param concatenate If true, appends the text; otherwise, replaces it.
   */
  void setPlainText(const QString &text, bool concatenate = false);

  /** @brief Increases the editor font size. */
  void zoomIn();

  /** @brief Decreases the editor font size. */
  void zoomOut();

  /** @brief Resets the editor font size to the default value. */
  void zoomReset();

private:
  /**
   * @brief Moves the current line(s) up or down.
   * @param up True to move up, false to move down.
   */
  void moveLineUpDown(bool up);

  /** @brief Pointer to the syntax highlighter engine. */
  GeneralHighlighter *generalHighlighter;

  /** @brief Flag checking if a completion popup was just used. */
  bool completedAndSelected;

  /**
   * @brief Checks if the 'Enter' or 'Tab' key handled a completion event.
   * @param event The key event.
   * @return True if handled, false otherwise.
   */
  bool handledCompletedAndSelected(QKeyEvent *event);

  /**
   * @brief Inserts paired characters intelligently (e.g., closing braces).
   * @param leftKey The opening character.
   * @param rightKey The closing character (optional).
   */
  void smartTextInsertion(QString leftKey, QString rightKey = QString());

  /** @brief Application settings manager. */
  DSettings settings;

  /** @brief Applies indentation logic when a new line is created. */
  void smartTextIndentation();

  /** @brief Flag to detect double Home key press (toggle start of line vs start of text). */
  bool homeKeyHitedTwice;

  /** @brief Initializes all QAction objects. */
  void createActions();

  /**
   * @brief Performs indentation or unindentation logic.
   * @param doIndent True to indent, false to unindent.
   */
  void indentOrUnindent(bool doIndent);

  /**
   * @brief Calculates the visual position of the indentation for a line.
   * @param text The line text.
   * @return The indentation width.
   */
  int lineIndentPosition(const QString &text) const;

  /**
   * @brief Finds the index of the first non-whitespace character.
   * @param text The text to search.
   * @return The index of the first non-space char.
   */
  int firstNonSpace(const QString &text) const;

  /**
   * @brief Extracts the indentation string (spaces/tabs) from a line.
   * @param text The line text.
   * @return The string comprising only the leading whitespace.
   */
  QString indentationString(const QString &text) const;

  /**
   * @brief Calculates the column index for indentation purposes.
   * @param text The line text.
   * @return The column number.
   */
  int indentationColumn(const QString &text) const;

  /**
   * @brief Maps a string position to a visual column.
   * @param text The text context.
   * @param position The cursor position.
   * @return The visual column index.
   */
  int columnAt(const QString &text, int position) const;

  /**
   * @brief Calculates the target column after indentation/unindentation.
   * @param column The current column.
   * @param doIndent True to add indent, false to remove.
   * @return The new column index.
   */
  int indentedColumn(int column, bool doIndent) const;

  /**
   * @brief Counts the number of spaces to the left of a position.
   * @param text The text context.
   * @param position The starting position.
   * @return The number of spaces.
   */
  int spacesLeftFromPosition(const QString &text, int position) const;

  /**
   * @brief Generates an indentation string to reach a target column.
   * @param startColumn The starting column.
   * @param targetColumn The desired target column.
   * @param block The text block for context (tab settings).
   * @return The indentation string.
   */
  QString indentationString(int startColumn, int targetColumn, const QTextBlock &block) const;

  /**
   * @brief Guesses if the block uses spaces or tabs for indentation.
   * @param _block The block to analyze.
   * @return True if it appears to use spaces.
   */
  bool guessSpacesForTabs(const QTextBlock &_block) const;

  /**
   * @brief Draws the folding markers (triangles/plus/minus) in the margin.
   * @param block The current block.
   * @param painter The painter object.
   * @param top The top Y-coordinate.
   */
  void drawFoldMarks(const QTextBlock &block, QPainter &painter, const int &top);

  /**
   * @brief Draws indicators for modified lines (usually green/yellow bars).
   * @param block The current block.
   * @param painter The painter object.
   * @param top The top Y-coordinate.
   */
  void drawModifiedLines(const QTextBlock &block, QPainter &painter, const int &top);

  /** @brief The type of editor currently active. */
  EditorTypes::EditorType editorType;

  /** @brief Global flag indicating if the document contains folded regions. */
  bool folded;

  /** @brief Refreshes highlighting based on mouse interaction (e.g., hovering). */
  void rehighlightOnMouse();

  /** @brief Pointer to the system clipboard. */
  QClipboard *clipboard;

  /** @brief Helper object for text folding logic. */
  FoldingTextHandler *foldingTextHandler;

  /**
   * @brief Recursive helper to match right brackets.
   * @param currentBlock The starting block.
   * @param index The current index.
   * @param numRightBracket Counter for nested brackets.
   * @param rightBracket The character to match.
   * @param firstTime Flag for the initial call.
   * @return True if matched.
   */
  bool matchRightBracket(QTextBlock currentBlock, int index, int numRightBracket, const char rightBracket, bool firstTime = false);

  /**
   * @brief Recursive helper to match left brackets.
   * @param currentBlock The starting block.
   * @param index The current index.
   * @param numLeftBracket Counter for nested brackets.
   * @param leftBracket The character to match.
   * @return True if matched.
   */
  bool matchLeftBracket(QTextBlock currentBlock, int index, int numLeftBracket, const char leftBracket);

  /**
   * @brief Creates a selection between matching brackets.
   * @param pos The cursor position.
   */
  void createBracketSelection(int pos);

  /** @brief Helper object for framing text logic. */
  FramingTextHandler *framingTextHandler;

  /**
   * @brief Calculates the marker size for the sidebar.
   * @param top The top Y-coordinate.
   * @return The bounding rectangle.
   */
  inline QRectF markerSize(int top);

  /** @brief Custom font applied to the editor. */
  QFont newFont;

  /**
   * @brief Applies the editor font to child widgets (like line numbers).
   * @param font The font to apply.
   */
  void setFontToWidgets(const QFont &font);

  /**
   * @brief Handles autocompletion triggers when a period is typed.
   * @param cursor The current cursor position.
   */
  void periodCompletionText(QTextCursor cursor);

signals:
  /** @brief Emitted when the ESC key is pressed. */
  void escKeyPressed();

  /**
   * @brief Emitted to request period-based completion.
   * @param completionPrefix The text prefix triggering completion.
   */
  void performPeriodCompletion(QString completionPrefix);

  /**
   * @brief Emitted when a file is requested to be opened.
   * @param fileName The path to the file.
   */
  void openFile(QString fileName);
};

/**
 * @brief The LineNumberArea class renders line numbers for the BaseTextEditor.
 */
class LineNumberArea : public QWidget
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the LineNumberArea.
   * @param editor The BaseTextEditor instance to attach to.
   */
  LineNumberArea(BaseTextEditor *editor);

protected:
  /**
   * @brief Paints the line numbers.
   * @param event The paint event parameters.
   */
  void paintEvent(QPaintEvent *event);

  /**
   * @brief Handles double-click events in the margin (e.g., selecting a line).
   * @param event The mouse event parameters.
   */
  void mouseDoubleClickEvent(QMouseEvent *event);

  /**
   * @brief Handles mouse press events in the margin.
   * @param event The mouse event parameters.
   */
  void mousePressEvent(QMouseEvent *event);

  /**
   * @brief General event filter.
   * @param event The event to process.
   * @return True if the event was handled.
   */
  bool event(QEvent *event);

private:
  /** @brief Reference to the parent code editor. */
  BaseTextEditor *codeEditor;

  /**
   * @brief Handles hover events for tooltips or effects.
   * @param event The hover event parameters.
   */
  void hoverEventHandler(QHoverEvent *event);
};

#endif // BASETEXTEDITOR_H
