/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QPainter>
#include <QTextBlock>
#include <QAction>
#include <QClipboard>
#include <QApplication>
#include <QFontMetrics>
#include <QTextDocumentFragment>
#include <QPainterPath>
#include <QRegularExpression>
#include <QMimeData>

#include "basetexteditor.h"
#include "generalhighlighter.h"
#include "staticfunctions.h"
#include "stdlib.h"
#include "foldingtexthandler.h"
#include "framingtexthandler.h"
#include "dicon.h"

#include <QDebug>

BaseTextEditor::BaseTextEditor(EditorTypes::EditorType type)
{
  folded = false;
  generalHighlighter = new GeneralHighlighter(document(), type);
  editorType = type;
  showNumberArea = true;
  homeKeyHitedTwice = true;
  foldRowList = new QVector<unsigned int>;
  foldedRowList = new QMap<unsigned int, QVector<unsigned int>>;
  modifiedRowList = new QVector<unsigned int>;
  lineNumberArea = new LineNumberArea(this);
  lineNumberArea->setGeometry(0, 0, 0, 0);

  newFont = StaticFunctions::fixedWidthFont();
  newFont.setPointSize(settings.value("TextEditor/PointSize", 12).toInt());
  setFontToWidgets(newFont);
  connect(this, SIGNAL(blockCountChanged(int)), this, SLOT(updateLineNumberAreaWidth()));
  connect(this, SIGNAL(updateRequest(QRect,int)), this, SLOT(updateLineNumberArea(QRect,int)));
  connect(this, SIGNAL(cursorPositionChanged()), this, SLOT(highlightCurrentLine()));
  connect(this, SIGNAL(textChanged()), this, SLOT(textChangedSlot()));
  updateLineNumberAreaWidth();
  if (settings.value("TextEditor/VisualizeSpaces", false).toBool()) {
    QTextOption option =  document()->defaultTextOption();
    option.setFlags(option.flags() | QTextOption::ShowTabsAndSpaces);
    document()->setDefaultTextOption(option);
  }
  pointList = QStringList();
  foldingTextHandler = new FoldingTextHandler;
  document()->documentLayout()->registerHandler(FoldingTextHandler::UserDefinedTextFormatObjectType, foldingTextHandler);
  framingTextHandler = new FramingTextHandler;
  document()->documentLayout()->registerHandler(FramingTextHandler::UserDefinedTextFormatObjectType, framingTextHandler);
  createActions();
  retranslateUI();
}

void BaseTextEditor::zoomIn()
{
  newFont = font();
  newFont.setPointSize(font().pointSize() + 1);
//  settings.setValue("TextEditor/PointSize", newFont.pointSize());
  setFontToWidgets(newFont);
}

void BaseTextEditor::zoomOut()
{
  newFont = font();
  newFont.setPointSize(font().pointSize() - 1);
//  settings.setValue("TextEditor/PointSize", newFont.pointSize());
  setFontToWidgets(newFont);
}

void BaseTextEditor::zoomReset()
{
  newFont = font();
  newFont.setPointSize(settings.value("TextEditor/DefaultPointSize", 11).toInt());
//  settings.setValue("TextEditor/PointSize", settings.value("TextEditor/DefaultPointSize", 11).toInt());
  setFontToWidgets(newFont);
}

void BaseTextEditor::setFontToWidgets(const QFont &)
{
  setFont(newFont);
  lineNumberArea->setFont(newFont);
}

void BaseTextEditor::showHideLineNumbers(bool show)
{
  showNumberArea = show;
  lineNumberArea->setVisible(showNumberArea);
  if (!showNumberArea)
    lineNumberArea->hide();
  updateLineNumberAreaWidth();
}

void BaseTextEditor::dockmarkCurrentLine(const QPoint &pos)
{
  QTextCursor textCursor = cursorForPosition(pos);
  // Variable used due to sign comparation
  uint bNumber = textCursor.blockNumber();
  if (!(foldRowList->contains(bNumber)
    ||
    foldedRowList->contains(bNumber)
    )) {
    QString blockNumber = QString("%1").arg(textCursor.blockNumber());
    if (pointList.contains(blockNumber)) {
      int valueToRemove = pointList.indexOf(blockNumber);
      if (valueToRemove >= 0)
        pointList.removeAt(valueToRemove);
    } else {
      pointList.append(blockNumber);
    }
    repaint();
  }
}

void BaseTextEditor::updateLineNumberAreaWidth()
{
  if (showNumberArea)
    setViewportMargins(lineNumberAreaWidth(), 0, 0, 0);
  else
    setViewportMargins(0, 0, 0, 0);
}

void BaseTextEditor::updateLineNumberArea(const QRect &rect, int dy)
{
  if (dy) {
    lineNumberArea->scroll(0, dy);
  } else {
    lineNumberArea->update(0, rect.y(), lineNumberArea->width(), rect.height());
  }
  if (rect.contains(viewport()->rect()))
    updateLineNumberAreaWidth();
}

void BaseTextEditor::highlightCurrentLine()
{
  if (!isReadOnly()) {
    QList<QTextEdit::ExtraSelection> extraSelections;
    QTextEdit::ExtraSelection selection;
    selection.format.setBackground(QColor(229, 233, 243));
    selection.format.setProperty(QTextFormat::FullWidthSelection, true);
    selection.cursor = textCursor();
    selection.cursor.clearSelection();
    extraSelections.append(selection);
    setExtraSelections(extraSelections);
  }
}

void BaseTextEditor::resizeEvent(QResizeEvent *event)
{
  QPlainTextEdit::resizeEvent(event);
  if (showNumberArea)
    lineNumberArea->setGeometry(QRect(contentsRect().left(), contentsRect().top(), lineNumberAreaWidth(), contentsRect().height()));
}

void BaseTextEditor::keyPressEvent(QKeyEvent *event)
{
  if (completedAndSelected && handledCompletedAndSelected(event))
    return;
  completedAndSelected = false;
  switch (event->key()) {
  case Qt::Key_Escape:
    emit escKeyPressed();
    break;
  }
//  if (completer->popup()->isVisible()) {
//    switch (event->key()) {
//    case Qt::Key_Up:
//    case Qt::Key_Down:
//    case Qt::Key_Enter:
//    case Qt::Key_Return:
//    case Qt::Key_Escape:
//      event->ignore();
//      return;
//    default:
//      completer->popup()->hide();
//      break;
//    }
//  }

  QTextCursor cursor;
  Qt::KeyboardModifiers modifiersControlShift;
  switch (event->key()) {
  case Qt::Key_AltGr:
  case Qt::Key_F3:
    //Qt does no handle AltGrModifiers
    //Qt::ALT Qt::AltModifier The normal Alt keys, but not keys like AltGr.
    //http://doc.qt.io/qt-5/qt.html
    //Qt::Key_QuoteLeft
    smartTextInsertion("`", "`");
    return;
  case Qt::Key_Backspace:
    cursor = textCursor();
    cursor.select(QTextCursor::WordUnderCursor);
    if (cursor.selectedText() == "()") {
      setTextCursor(cursor);
    }
    if (cursor.selectedText().right(2) == "()" && cursor.selectedText().length() > 2) {
      cursor.insertText(cursor.selectedText().left(cursor.selectedText().length() - 2).repeated(2));
      setTextCursor(cursor);
    }
    break;
  case Qt::Key_QuoteDbl:
    smartTextInsertion("\"");
    return;
  case Qt::Key_Apostrophe:
    smartTextInsertion("'");
    return;
  case Qt::Key_Percent:
    cursor = textCursor();
    cursor.movePosition(QTextCursor::PreviousCharacter, QTextCursor::MoveAnchor);
    cursor.select(QTextCursor::WordUnderCursor);
    if (cursor.selectedText() == "''" || cursor.selectedText() == "\"\"") {
      smartTextInsertion("%");
    } else {
      QPlainTextEdit::keyPressEvent(event);
    }
    return;
  case Qt::Key_Space:
    if (settings.value("GeneralSettings/UppercaseKeywordsAndReservedWords", false).toBool()) {
      cursor = textCursor();
      cursor.movePosition(QTextCursor::PreviousCharacter, QTextCursor::MoveAnchor);
      cursor.select(QTextCursor::WordUnderCursor);
      if (StaticFunctions::mariadbKeywords().contains(cursor.selectedText(), Qt::CaseInsensitive)) {
        //      cursor.removeSelectedText();
        cursor.insertText(cursor.selectedText().toUpper() + " ");
        setTextCursor(cursor);
      } else {
        QPlainTextEdit::keyPressEvent(event);
      }
    } else {
      QPlainTextEdit::keyPressEvent(event);
    }
    return;
  case Qt::Key_Agrave:
    smartTextInsertion("`");
    return;
  case Qt::Key_ParenLeft:
    smartTextInsertion("(", ")");
    return;
  case Qt::Key_BracketLeft:
    smartTextInsertion("[", "]");
    return;
  case Qt::Key_BraceLeft:
    smartTextInsertion("{", "}");
    return;
  case Qt::Key_Tab:
    indentAction->trigger();
    return;
//  case Qt::Key_Escape:
//    setFocusToEditorActionTriggered();
//    break;
  case Qt::Key_Home:
    homeKeyHitedTwice = !homeKeyHitedTwice;
    if (homeKeyHitedTwice) {
      QTextCursor cursor = textCursor();
      cursor.movePosition(QTextCursor::NextWord, QTextCursor::MoveAnchor);
      setTextCursor(cursor);
      return;
    }
    break;
    //  case Qt::Key_Delete:
    //    if (event->modifiers() == Qt::ControlModifier)
    //      cutLineActionTriggerd();
    //    break;
  case Qt::Key_Up:
  case Qt::Key_Down:
    if (event->modifiers() == Qt::AltModifier) {
      if (event->key() == Qt::Key_Up) {
        moveLineUpDown(true);
        break;
      } else if (event->key() == Qt::Key_Down) {
        moveLineUpDown(false);
        break;
      }
    }
    if (event->modifiers() == Qt::ControlModifier) {
      QList<int> list;
      QTextCursor cursor = textCursor();
      for (int counter = 0; counter < pointList.count(); counter++)
        list.append(pointList.at(counter).toUInt());
      if (list.indexOf(cursor.blockNumber()) == -1)
        list.append(cursor.blockNumber());
      std::sort(list.begin(), list.end());
      if ((cursor.blockNumber() == list.last() && event->key() == Qt::Key_Down)
          || (cursor.blockNumber() == list.first() && event->key() == Qt::Key_Up))
        break;
      if (event->key() == Qt::Key_Up) {
        gotoLine(list.at(list.indexOf(cursor.blockNumber()) - 1));
        break;
      } else if (event->key() == Qt::Key_Down) {
        gotoLine(list.at(list.indexOf(cursor.blockNumber()) + 1));
        break;
      }
    }
    modifiersControlShift |= Qt::ShiftModifier;
    modifiersControlShift |= Qt::ControlModifier;
    if (event->modifiers() == modifiersControlShift) {
      QTextCursor cursor = textCursor();
      cursor.setPosition(cursor.position());
      cursor.setPosition(document()->find(";", cursor.position()).position(), QTextCursor::KeepAnchor);
      setTextCursor(cursor);
    }
    break;
  case Qt::Key_Return:
  case Qt::Key_Enter:
    cursor = textCursor();
    cursor.movePosition(QTextCursor::StartOfLine);
    cursor.select(QTextCursor::LineUnderCursor);
    if (cursor.selectedText().startsWith("/***"))
      cursor.insertText("/**********************************************************************************************/");
    break;
  case Qt::Key_Period:
    periodCompletionText(textCursor());
    break;
  case Qt::Key_V:
      if (event->modifiers() == Qt::ControlModifier) {
        cursor = textCursor();
        clipboard = QApplication::clipboard();
        cursor.insertText(clipboard->text());
        if (settings.value("TextEditor/SelectOnPaste", Qt::Unchecked) == Qt::Checked)
          cursor.select(QTextCursor::BlockUnderCursor);
        setTextCursor(cursor);
      }
      break;
  default:
    homeKeyHitedTwice = true;
  }
  if (!(event->key() == Qt::Key_V && event->modifiers() == Qt::ControlModifier)) {
    QPlainTextEdit::keyPressEvent(event);
  }

  if (settings.value("TextEditor/AutomaticIndentation", Qt::Unchecked) == Qt::Checked)
    switch (event->key()) {
    case Qt::Key_Return:
    case Qt::Key_Enter:
      smartTextIndentation();
      break;
    // default: Q_ASSERT(false);
    }
  //  QPlainTextEdit::keyPressEvent(event);
}

void BaseTextEditor::periodCompletionText(QTextCursor cursor) {
  if (cursor.movePosition(QTextCursor::WordLeft, QTextCursor::KeepAnchor)) {
    static QRegularExpression regularExpression("`.*`", QRegularExpression::CaseInsensitiveOption);
    if (cursor.selectedText().contains(regularExpression)) {
      // qDebug() << cursor.selectedText().mid(1, cursor.selectedText().length() - 2);
      emit performPeriodCompletion(cursor.selectedText().mid(1, cursor.selectedText().length() - 2));
    } else {
      periodCompletionText(cursor);
    }
  }
}

void BaseTextEditor::moveLineUpDown(bool up)
{
  QTextCursor cursor = textCursor();
  QTextCursor move = cursor;

  move.setVisualNavigation(false); // this opens folded items instead of destroying them
  move.beginEditBlock();

  bool hasSelection = cursor.hasSelection();

  if (hasSelection) {
    move.setPosition(cursor.selectionStart());
    move.movePosition(QTextCursor::StartOfBlock);
    move.setPosition(cursor.selectionEnd(), QTextCursor::KeepAnchor);
    move.movePosition(move.atBlockStart() ? QTextCursor::PreviousCharacter: QTextCursor::EndOfBlock, QTextCursor::KeepAnchor);
  } else {
    move.movePosition(QTextCursor::StartOfBlock);
    move.movePosition(QTextCursor::EndOfBlock, QTextCursor::KeepAnchor);
  }
  QString text = move.selectedText();

  move.movePosition(QTextCursor::NextCharacter, QTextCursor::KeepAnchor);
  move.removeSelectedText();

  if (up) {
    move.movePosition(QTextCursor::PreviousBlock);
    move.insertBlock();
    move.movePosition(QTextCursor::PreviousCharacter);
  } else {
    move.movePosition(QTextCursor::EndOfBlock);
    if (move.atBlockStart()) { // empty block
      move.movePosition(QTextCursor::NextBlock);
      move.insertBlock();
      move.movePosition(QTextCursor::PreviousCharacter);
    } else {
      move.insertBlock();
    }
  }

  int start = move.position();
  move.clearSelection();
  move.insertText(text);
  int end = move.position();

  if (hasSelection) {
    move.setPosition(end);
    move.setPosition(start, QTextCursor::KeepAnchor);
  } else {
    move.setPosition(start);
  }
  move.endEditBlock();
  setTextCursor(move);
}

void BaseTextEditor::mouseDoubleClickEvent(QMouseEvent *event)
{
  QPlainTextEdit::mouseDoubleClickEvent(event);
  rehighlightOnMouse();
}

//void BaseTextEditor::mouseMoveEvent(QMouseEvent *event)
//{
//  QPlainTextEdit::mouseMoveEvent(event);
//  rehighlightOnMouse();
//}

void BaseTextEditor::dropEvent(QDropEvent *e)
{
#if defined(Q_OS_WIN)
  emit openFile(e->mimeData()->text().trimmed().mid(8));
#else
  emit openFile(e->mimeData()->text().trimmed().mid(7));
#endif
}

bool BaseTextEditor::handledCompletedAndSelected(QKeyEvent *event)
{
  completedAndSelected = false;
  QTextCursor cursor = textCursor();
  switch (event->key()) {
  case Qt::Key_Enter:
  case Qt::Key_Return:
    cursor.clearSelection();
    break;
  case Qt::Key_Escape:
    cursor.removeSelectedText();
    break;
  default: return false;
  }
  setTextCursor(cursor);
  event->accept();
  return true;
}

void BaseTextEditor::smartTextInsertion(QString leftKey, QString rightKey)
{
  QTextCursor cursor = textCursor();
  bool hasSelection = cursor.hasSelection();
  cursor.insertText(leftKey + cursor.selectedText() + (rightKey.isEmpty() ? leftKey : rightKey));
  if (!hasSelection)
    cursor.movePosition(QTextCursor::PreviousCharacter, QTextCursor::MoveAnchor);
  setTextCursor(cursor);
}

void BaseTextEditor::smartTextIndentation()
{
  QString prevText(textCursor().block().previous().text());
  if (!prevText.isEmpty()) {
    int count = 0;
    for (int pos = 0; pos <= prevText
         .length(); pos++)
      if (prevText[pos] == QChar(' '))
        count++;
      else
        break;
    QTextCursor cursor = textCursor();
    cursor.insertText(QString(" ").repeated(count));
    setTextCursor(cursor);
  }
}

void BaseTextEditor::createActions()
{
  indentAction = new QAction(this);
  indentAction->setIcon(DIcon::FormatIndentMore());
  indentAction->setShortcut(QKeySequence(Qt::Key_Tab));
  connect(indentAction, SIGNAL(triggered()), this, SLOT(indentActionTriggered()));
  unindentAction = new QAction(this);
  unindentAction->setIcon(DIcon::FormatIndentLess());
  unindentAction->setShortcut(QKeySequence(Qt::SHIFT | Qt::Key_Tab));
  connect(unindentAction, SIGNAL(triggered()), this, SLOT(unindentActionTriggered()));
  insertLicenceTemplateAction = new QAction(this);
  connect(insertLicenceTemplateAction, SIGNAL(triggered()), this, SLOT(insertLicenceTemplateSlot()));
  visualizeSpacesAction = new QAction(this);
  visualizeSpacesAction->setCheckable(true);
  visualizeSpacesAction->setChecked(settings.value("TextEditor/VisualizeSpaces", false).toBool());
  connect(visualizeSpacesAction, SIGNAL(toggled(bool)), this, SLOT(visualizeSpacesActionTriggered(bool)));
  foldAction = new QAction(this);
  connect(foldAction, SIGNAL(triggered()), this, SLOT(foldActionTriggered()));
  unfoldAction = new QAction(this);
  connect(unfoldAction, SIGNAL(triggered()), this, SLOT(unfoldActionTriggered()));
}

void BaseTextEditor::retranslateUI()
{
  foldAction->setText(tr("Fold text"));
  foldAction->setToolTip(foldAction->text());
  unfoldAction->setText(tr("Unfold text"));
  unfoldAction->setToolTip(unfoldAction->text());
  indentAction->setText(tr("Indent"));
  indentAction->setToolTip(indentAction->text());
  unindentAction->setText(tr("Unindent"));
  unindentAction->setToolTip(unindentAction->text());
  insertLicenceTemplateAction->setText(tr("Insert licence template"));
  insertLicenceTemplateAction->setToolTip(insertLicenceTemplateAction->text());
  visualizeSpacesAction->setText(tr("Visualize spaces"));
  visualizeSpacesAction->setToolTip(visualizeSpacesAction->text());
}

QStringList BaseTextEditor::wordList()
{
  static QRegularExpression re("\\W+");
  QStringList words = toPlainText().split(re);
  words.removeDuplicates();
  return words;
}

void BaseTextEditor::insertLicenceTemplateSlot()
{
  QTextCursor cursor = textCursor();
  cursor.insertText(settings.value("License/Template", "").toString());
  setTextCursor(cursor);
}

void BaseTextEditor::setPlainText(const QString &text, bool concatenate)
{
  if (concatenate) {
    QPlainTextEdit::setPlainText(text + "\n" + toPlainText());
  } else {
    QPlainTextEdit::setPlainText(text);
  }
}

void BaseTextEditor::indentActionTriggered()
{
  indentOrUnindent(true);
}

void BaseTextEditor::unindentActionTriggered()
{
  indentOrUnindent(false);
}

void BaseTextEditor::visualizeSpacesActionTriggered(bool toggled)
{
  QTextOption option = document()->defaultTextOption();
  if (toggled)
    option.setFlags(option.flags() | QTextOption::ShowTabsAndSpaces);
  else
    option.setFlags(option.flags() & ~QTextOption::ShowTabsAndSpaces);
  //option.setFlags(option.flags() | QTextOption::AddSpaceForLineAndParagraphSeparators);
  document()->setDefaultTextOption(option);
  settings.setValue("TextEditor/VisualizeSpaces", toggled);
}

Q_DECLARE_METATYPE(QTextDocumentFragment)

void BaseTextEditor::foldActionTriggered()
{
  QTextCursor cursor = textCursor();
  if (cursor.hasSelection()) {
    QTextCharFormat charFormat;
    charFormat.setObjectType(FoldingTextHandler::UserDefinedTextFormatObjectType);
    QVariant value;
    value.setValue(cursor.selection());
    charFormat.setProperty(FoldingTextHandler::UserDefinedCharFormatPropertyId, value);
    cursor.insertText(QString(QChar::ObjectReplacementCharacter), charFormat);
  }
}

void BaseTextEditor::unfoldActionTriggered()
{
  QTextCursor cursor = textCursor();
  if (!cursor.hasSelection()) {
    QTextCharFormat charFormat = cursor.charFormat();
    if (charFormat.objectType() == (FoldingTextHandler::UserDefinedTextFormatObjectType)) {
      cursor.movePosition(QTextCursor::Left, QTextCursor::KeepAnchor);
      QVariant value = charFormat.property(FoldingTextHandler::UserDefinedCharFormatPropertyId);
      QTextDocumentFragment documentFragment = value.value<QTextDocumentFragment>();
      cursor.insertFragment(documentFragment);
    }
  }
}

void BaseTextEditor::textChangedSlot()
{
  if (fileLineByLine.count() == 0 || lineFolded || textCursor().hasSelection()) {
    lineFolded = false;
    return;
  }
  if (textCursor().blockNumber() >= fileLineByLine.count()) {
    appendModifiedRowList(textCursor().blockNumber());
    return;
  }
  if (fileLineByLine.at(textCursor().blockNumber()).trimmed() != textCursor().block().text().trimmed()) {
    appendModifiedRowList(textCursor().blockNumber());
    return;
  }
}

void BaseTextEditor::indentOrUnindent(bool doIndent)
{
  QTextCursor cursor = textCursor();
  cursor.beginEditBlock();

  if (cursor.hasSelection()) {
    int pos = cursor.position();
    int anchor = cursor.anchor();
    int start = qMin(anchor, pos);
    int end = qMax(anchor, pos);

    QTextDocument *doc = document();
    QTextBlock startBlock = doc->findBlock(start);
    QTextBlock endBlock = doc->findBlock(end - 1).next();

    for (QTextBlock block = startBlock; block != endBlock; block = block.next()) {
      QString text(block.text());
      int indentPosition = lineIndentPosition(text);
      if (!doIndent && !indentPosition)
        indentPosition = firstNonSpace(text);
      int targetColumn = indentedColumn(columnAt(text, indentPosition), doIndent);
      cursor.setPosition(block.position() + indentPosition);
      cursor.insertText(indentationString(0, targetColumn, block));
      cursor.setPosition(block.position());
      cursor.setPosition(block.position() + indentPosition, QTextCursor::KeepAnchor);
      cursor.removeSelectedText();
    }
  } else {
    QTextBlock block = cursor.block();
    QString text(block.text());
    int indentPosition = cursor.position() - block.position();
    int spaces = spacesLeftFromPosition(text, indentPosition);
    int startColumn = columnAt(text, indentPosition - spaces);
    int targetColumn = indentedColumn(columnAt(text, indentPosition), doIndent);
    cursor.setPosition(block.position() + indentPosition);
    cursor.setPosition(block.position() + indentPosition - spaces, QTextCursor::KeepAnchor);
    cursor.removeSelectedText();
    cursor.insertText(indentationString(startColumn, targetColumn, block));
    setTextCursor(cursor);
  }
  cursor.endEditBlock();
}

int BaseTextEditor::lineIndentPosition(const QString &text) const
{
  int i = 0;
  while (i < text.size()) {
    if (!text.at(i).isSpace())
      break;
    ++i;
  }
  int column = columnAt(text, i);
  return i - (column % settings.value("TextEditor/TabSize", 2).toInt());
}

int BaseTextEditor::firstNonSpace(const QString &text) const
{
  int i = 0;
  while (i < text.size()) {
    if (!text.at(i).isSpace())
      return i;
    ++i;
  }
  return i;
}

QString BaseTextEditor::indentationString(const QString &text) const
{
  return text.left(firstNonSpace(text));
}

int BaseTextEditor::indentationColumn(const QString &text) const
{
  return columnAt(text, firstNonSpace(text));
}

int BaseTextEditor::columnAt(const QString &text, int position) const
{
  int column = 0;
  for (int i = 0; i < position; ++i) {
    if (text.at(i) == QLatin1Char('\t'))
      column = column - (column % settings.value("TextEditor/TabSize", 2).toInt()) + settings.value("TextEditor/TabSize", 2).toInt();
    else
      ++column;
  }
  return column;
}

int BaseTextEditor::indentedColumn(int column, bool doIndent) const
{
  int aligned = (column / settings.value("TextEditor/TabSize", 2).toInt()) * settings.value("TextEditor/TabSize", 2).toInt();
  if (doIndent)
    return aligned + settings.value("TextEditor/TabSize", 2).toInt();
  if (aligned < column)
    return aligned;
  return qMax(0, aligned - settings.value("TextEditor/TabSize", 2).toInt());
}

int BaseTextEditor::spacesLeftFromPosition(const QString &text, int position) const
{
  int pos = position;
  while (pos > 0) {
    if (!text.at(pos-1).isSpace())
      break;
    --pos;
  }
  return position - pos;
}

QString BaseTextEditor::indentationString(int startColumn, int targetColumn, const QTextBlock &block) const
{
  targetColumn = qMax(startColumn, targetColumn);
  if (guessSpacesForTabs(block))
    return QString(targetColumn - startColumn, QLatin1Char(' '));

  QString s;
  int alignedStart = startColumn - (startColumn % settings.value("TextEditor/TabSize", 2).toInt())
      + settings.value("TextEditor/TabSize", 2).toInt();
  if (alignedStart > startColumn && alignedStart <= targetColumn) {
    s += QLatin1Char('\t');
    startColumn = alignedStart;
  }
  if (int columns = targetColumn - startColumn) {
    int tabs = columns / settings.value("TextEditor/TabSize", 2).toInt();
    s += QString(tabs, QLatin1Char('\t'));
    s += QString(columns - tabs * settings.value("TextEditor/TabSize", 2).toInt(), QLatin1Char(' '));
  }
  return s;
}

bool BaseTextEditor::guessSpacesForTabs(const QTextBlock &block) const
{
  if (block.isValid()) {
    const QTextDocument *doc = block.document();
    QVector<QTextBlock> currentBlocks(2, block);
    int maxLookAround = 100;
    while (maxLookAround-- > 0) {
      if (currentBlocks.at(0).isValid())
        currentBlocks[0] = currentBlocks.at(0).previous();
      if (currentBlocks.at(1).isValid())
        currentBlocks[1] = currentBlocks.at(1).next();
      bool done = true;
      foreach (const QTextBlock &block, currentBlocks) {
        if (block.isValid())
          done = false;
        if (!block.isValid() || block.length() == 0)
          continue;
        const QChar firstChar = doc->characterAt(block.position());
        if (firstChar == QLatin1Char(' ')) {
          return true;
        } else if (firstChar == QLatin1Char('\t')) {
          return false;
        }
      }
      if (done)
        break;
    }
  }
  return true;
}

void BaseTextEditor::drawFoldMarks(const QTextBlock &block, QPainter &painter, const int &top)
{
  switch(editorType) {
  case EditorTypes::SQLQuery:
    painter.setOpacity(2);
    if ((block.text().startsWith("CREATE", Qt::CaseInsensitive) && block.text().endsWith("(", Qt::CaseInsensitive))
        || block.text() == QString(QChar::ObjectReplacementCharacter)) {
      if (foldedRowList->contains(block.blockNumber())) {
        painter.drawImage(markerSize(top), QImage(DIcon::GoNextIconPath()));
      } else {
        painter.drawImage(markerSize(top), QImage(DIcon::GoDownIconPath()));
        foldRowList->append(block.blockNumber());
      }
    }
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
  case EditorTypes::NoEditor:
    break;
  // default: Q_ASSERT(false);i
  }
}

void BaseTextEditor::rehighlightOnMouse()
{
  // // If mouse tracking is switched off, mouse move events only occur if a mouse button is pressed while the mouse is being moved. If mouse tracking is switched on, mouse move events occur even if no mouse button is pressed.
  // QTextCursor cursor = textCursor();
  // if (editorType != EditorTypes::NoEditor && cursor.selectedText().length() <= 64)
  //   rehighlight(cursor.selectedText());

  // Limpiamos selecciones previas que no sean la linea actual
  // Nota: Esto depende de cómo quieras gestionar la línea actual, ver punto 2.
  QList<QTextEdit::ExtraSelection> selections = extraSelections();

  // Filtramos para dejar solo la linea actual (si existe) y quitar los resaltados viejos
  // O simplemente empezamos una lista nueva si vas a regenerar la linea actual después.
  // Para simplificar, asumiremos que queremos mantener lo que ya hay y añadir lo nuevo:

  QTextCursor cursor = textCursor();
  QString selectedText = cursor.selectedText();

  // Validaciones básicas
  if (editorType == EditorTypes::NoEditor || selectedText.isEmpty() || selectedText.length() > 64) {
    return;
  }

  // Configuración del color de resaltado (amarillo suave, por ejemplo)
  QTextCharFormat format;
  // format.setBackground(QColor(255, 255, 200)); // Ajusta este color a tu tema
  format.setBackground(QBrush(QColor(255, 239, 11)));

  // Buscar todas las ocurrencias en el documento
  QTextDocument *doc = document();
  QTextCursor highlightCursor(doc);

  // Usamos FindWholeWords para que "int" no marque "interruptor"
  QTextDocument::FindFlags options = QTextDocument::FindWholeWords | QTextDocument::FindCaseSensitively;

  while (!highlightCursor.isNull() && !highlightCursor.atEnd()) {
    highlightCursor = doc->find(selectedText, highlightCursor, options);
    if (!highlightCursor.isNull()) {
      QTextEdit::ExtraSelection extra;
      extra.format = format;
      extra.cursor = highlightCursor;
      selections.append(extra);
    }
  }
  setExtraSelections(selections);
}

int BaseTextEditor::lineNumberAreaWidth()
{
  int digits = 1;
  int max = qMax(1, blockCount());
  while (max >= 10) {
    max /= 10;
    ++digits;
  }
  QFontMetrics metrics(font());
  return (metrics.horizontalAdvance(QLatin1Char('9')) * digits) + markerSize(0).width();
}

void BaseTextEditor::rehighlight(const QString &exp)
{
  generalHighlighter->setFindPattern(exp);
  generalHighlighter->rehighlight();
}

void BaseTextEditor::gotoLine(unsigned int lineNumber, bool center)
{
  QTextCursor cursor(document()->findBlockByNumber(lineNumber));
  cursor.movePosition(QTextCursor::Right, QTextCursor::MoveAnchor, 0);
  setTextCursor(cursor);
  if (center)
    centerCursor();
  setFocus();
}

void BaseTextEditor::lineNumberAreaPaintEvent(QPaintEvent *event)
{
  QPainter painter(lineNumberArea);
  foldRowList->clear();
  painter.fillRect(event->rect(), QColor(240, 244, 247));
  QTextBlock block = firstVisibleBlock();
  int blockNumber = block.blockNumber();
  int top = (int)blockBoundingGeometry(block).translated(contentOffset()).top();
  int bottom = top + (int)blockBoundingRect(block).height();
  while (block.isValid() && top <= event->rect().bottom()) {
    if (block.isVisible() && bottom >= event->rect().top()) {
      painter.drawText(0, top, lineNumberArea->width(), fontMetrics().height(), Qt::AlignRight, QString::number(blockNumber + 1));
      drawModifiedLines(block, painter, top);
      if (pointList.contains(QString("%1").arg(blockNumber)))
        painter.drawImage(markerSize(top), QImage(DIcon::DatabaseIconPath()));
      drawFoldMarks(block, painter, top);
//      qDebug() << ": "  << blockNumber - 1;
    }
    block = block.next();
    top = bottom;
    bottom = top + (int)blockBoundingRect(block).height();
    ++blockNumber;
  }
}

void BaseTextEditor::drawModifiedLines(const QTextBlock &block, QPainter &painter, const int &top)
{
  switch(editorType) {
  case EditorTypes::SQLQuery:
    if (modifiedRowList->contains((uint)block.blockNumber())) {
      painter.setRenderHint(QPainter::Antialiasing);
      QPainterPath path;
      path.addRoundedRect(markerSize(top), 4, 4);
      painter.fillPath(path, QColor(144, 238, 144));
      painter.drawPath(path);
    }
    break;
  case EditorTypes::Diff:
  case EditorTypes::Commit:
  case EditorTypes::SVNLog:
  case EditorTypes::NoEditor:
    break;
  // default: Q_ASSERT(false);
  }
}

QVector<unsigned int> BaseTextEditor::matchBracket()
{
  //QList<QTextEdit::ExtraSelection> selections = textEditor->extraSelections();
  //textEditor->setExtraSelections(selections);
  absolutePostionForTheLastBracketMatched.clear();
  TextBlockData *data = static_cast<TextBlockData *>(textCursor().block().userData());
  if (data) {
    QVector<BracketInfo *> infos = data->brackets();
    int pos = textCursor().block().position();
    for (int counter = 0; counter < infos.size(); ++counter) {
      foreach (char leftBracket, StaticFunctions::bracketLeftList()) {
        char rightBracket = StaticFunctions::rightBracketMatch(leftBracket);
        BracketInfo *info = infos.at(counter);
        int curPos = textCursor().position() - textCursor().block().position();
        if (info->position == curPos - 1 && info->character == leftBracket) {
          if (matchLeftBracket(textCursor().block(), counter + 1, 0, leftBracket)) {
            ///qDebug() << "Antes 1" << pos + info->position;
            createBracketSelection(pos + info->position);
          }
        } else if (info->position == curPos - 1 && info->character == rightBracket) {
          if (matchRightBracket(textCursor().block(), counter - 1, 0, rightBracket, true)) {
            ///qDebug() << "Antes 2" << pos + info->position;
            createBracketSelection(pos + info->position);
          }
        }
      }
    }
  }
//  qDebug() << absolutePostionForTheLastBracketMatched;
  std::sort(absolutePostionForTheLastBracketMatched.begin(), absolutePostionForTheLastBracketMatched.end());
  return absolutePostionForTheLastBracketMatched;
}

bool BaseTextEditor::matchLeftBracket(QTextBlock currentBlock, int index, int numLeftBracket, const char leftBracket)
{
  TextBlockData *data = static_cast<TextBlockData *>(currentBlock.userData());
  QVector<BracketInfo *> infos = data->brackets();
  int docPos = currentBlock.position();
  for (; index < infos.size(); ++index) {
    BracketInfo *info = infos.at(index);
    if (info->character == leftBracket) {
      ++numLeftBracket;
      continue;
    }
    if (info->character == StaticFunctions::rightBracketMatch(leftBracket) && numLeftBracket == 0) {
      ///qDebug() << "Antes 4" << docPos + info->position;
      createBracketSelection(docPos + info->position);
      return true;
    } else {
      --numLeftBracket;
    }
  }
  currentBlock = currentBlock.next();
  if (currentBlock.isValid())
    return matchLeftBracket(currentBlock, 0, numLeftBracket, leftBracket);
  return false;
}

void BaseTextEditor::createBracketSelection(int pos)
{
  ///qDebug() << "Después 3" << pos;
  absolutePostionForTheLastBracketMatched.append(pos);
  QList<QTextEdit::ExtraSelection> selections = extraSelections();
  QTextEdit::ExtraSelection selection;
  QTextCharFormat format = selection.format;
  format.setBackground(Qt::lightGray);
  selection.format = format;
  QTextCursor cursor = textCursor();
  cursor.setPosition(pos);
  cursor.movePosition(QTextCursor::NextCharacter, QTextCursor::KeepAnchor);
  selection.cursor = cursor;
  selections.append(selection);
  setExtraSelections(selections);
}

bool BaseTextEditor::matchRightBracket(QTextBlock currentBlock, int index, int numRightBracket, const char rightBracket, bool firstTime)
{
  TextBlockData *data = static_cast<TextBlockData *>(currentBlock.userData());
  QVector<BracketInfo *> parentheses = data->brackets();

  if(!firstTime) //Set i to the end of the parentheses list.
    index = parentheses.size() - 1;
  int docPos = currentBlock.position();
  for (; index > -1 && parentheses.size() > 0; --index) {
    BracketInfo *info = parentheses.at(index);
    if (info->character == rightBracket) {
      ++numRightBracket;
      continue;
    }
    if (info->character == StaticFunctions::leftBracketMatch(rightBracket) && numRightBracket == 0) {
      ///qDebug() << "Antes 5" << docPos + info->position;
      createBracketSelection(docPos + info->position);
      return true;
    } else {
      --numRightBracket;
    }
  }
  currentBlock = currentBlock.previous();
  if (currentBlock.isValid())
    return matchRightBracket(currentBlock, 0, numRightBracket, rightBracket);
  return false;
}

QRectF BaseTextEditor::markerSize(int top)
{
  qreal pointSize = settings.value("TextEditor/PointSize", settings.value("TextEditor/DefaultPointSize", 11).toInt()).toInt();
  QFont font;
  font.setPointSize(pointSize);
  QFontMetrics fontMetrics(font);
  return QRectF(0, top, fontMetrics.height(), fontMetrics.height());
}

void BaseTextEditor::appendModifiedRowList(unsigned int item)
{
  modifiedRowList->append(item);
//   modifiedRowList->erase(std::unique(modifiedRowList->begin(), modifiedRowList->end()), modifiedRowList->cend());
  modifiedRowList->erase(std::unique(modifiedRowList->begin(), modifiedRowList->end()), modifiedRowList->cend());
  std::sort(modifiedRowList->begin(), modifiedRowList->end());
}

void BaseTextEditor::setFontWeight(QFont::Weight weight)
{
  QFont newFont = font();
  newFont.setWeight(weight);
  setFont(newFont);
}

LineNumberArea::LineNumberArea(BaseTextEditor *editor) : QWidget(editor), codeEditor(editor)
{
  setMouseTracking(true);
  setAttribute(Qt::WA_Hover);
}

void LineNumberArea::paintEvent(QPaintEvent *event)
{
  codeEditor->lineNumberAreaPaintEvent(event);
}

void LineNumberArea::mouseDoubleClickEvent(QMouseEvent *event)
{
  codeEditor->dockmarkCurrentLine(event->pos());
  QWidget::mouseDoubleClickEvent(event);
}

void LineNumberArea::mousePressEvent(QMouseEvent *event)
{
  QTextCursor textCursor = codeEditor->cursorForPosition(event->pos());
  if (codeEditor->foldRowList->contains((uint)textCursor.blockNumber())
      || codeEditor->foldedRowList->contains((uint)textCursor.blockNumber())) {
    bool hide;
    codeEditor->lineFolded = true;
    QVector<unsigned int> values;
    unsigned int lineToGo = textCursor.blockNumber();
    if (codeEditor->foldedRowList->contains((uint)textCursor.blockNumber())) {
      hide = false;
      values = codeEditor->foldedRowList->take((uint)textCursor.blockNumber());
    } else {
      hide = true;
      textCursor.movePosition(QTextCursor::EndOfBlock);
      codeEditor->setTextCursor(textCursor);
      values = codeEditor->matchBracket();
      codeEditor->foldedRowList->insert(textCursor.blockNumber(), values);
    }
    QTextCharFormat charFormat;
    textCursor = codeEditor->cursorForPosition(event->pos());
    textCursor.select(QTextCursor::LineUnderCursor);
    if (hide) {
      charFormat.setObjectType(FramingTextHandler::UserDefinedTextFormatObjectType);
      charFormat.setProperty(FramingTextHandler::UserDefinedCharFormatPropertyId, textCursor.selectedText());
      textCursor.insertText(QString(QChar::ObjectReplacementCharacter), charFormat);
    } else {
      charFormat = textCursor.charFormat();
      textCursor.insertText(charFormat.property(FramingTextHandler::UserDefinedCharFormatPropertyId).toString());
      codeEditor->setTextCursor(textCursor);
    }
    QTextBlock block;
    if (values.count() > 0) {
      for (uint counter = values.at(0); counter <= values.at(1); counter++) {
        textCursor.setPosition(counter);
       block = textCursor.block();
       block.setVisible(!hide);
      }
    }
    codeEditor->viewport()->repaint();
    //repaint(); //Is not needed since FramingTextHandler invokes it
    codeEditor->gotoLine(lineToGo, false);
  }
  QWidget::mousePressEvent(event);
}

bool LineNumberArea::event(QEvent *event)
{
  if (event->type() == QEvent::HoverMove)
    hoverEventHandler(static_cast<QHoverEvent*>(event));
  return QWidget::event(event);
}

void LineNumberArea::hoverEventHandler(QHoverEvent *event)
{
  QTextCursor textCursor = codeEditor->cursorForPosition(event->position().toPoint());
  //  textCursor.blockNumber();
  QVector<unsigned int> values;
  QList<QTextEdit::ExtraSelection> extraSelections;
  QTextEdit::ExtraSelection selection;
  switch(event->type())
  {
  //  case QEvent::HoverEnter:
  //      hoverEnter(static_cast<QHoverEvent*>(event));
  //      return true;
  //      break;
  case QEvent::HoverMove:
    selection.format.setBackground(QColor(220, 220, 220));
    selection.format.setProperty(QTextFormat::FullWidthSelection, true);
    selection.cursor = textCursor;
    selection.cursor.clearSelection();
    extraSelections.append(selection);
    if (codeEditor->foldRowList->contains((uint)textCursor.blockNumber())) {
      textCursor.movePosition(QTextCursor::EndOfBlock);
      codeEditor->setTextCursor(textCursor);
      values = codeEditor->matchBracket();
      if (values.count() > 0) {
        for (uint counter = values.at(0); counter <= values.at(1); counter++) {
          textCursor.setPosition(counter);
          selection.cursor = textCursor;
          selection.cursor.clearSelection();
          extraSelections.append(selection);
        }
      }
    }
    codeEditor->setExtraSelections(extraSelections);
    break;
//  case QEvent::HoverLeave:
//    //codeEditor->viewport()->repaint();
//    break;
  default:
    break;
  }
}
