/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef BACKUP_H
#define BACKUP_H

class DTitleLabel;
class QTreeWidget;
class QGroupBox;
class BaseTextEditor;
class QTreeWidgetItem;
class QCheckBox;
class DFileSelector;
class QVBoxLayout;
class QPushButton;
class QProgressDialog;

#include "dbms.h"
#include "dmdisubwindow.h"

/**
 * @brief The Backup class manages the database backup interface and operations.
 *
 * This class inherits from DMdiSubWindow and provides a GUI for users to select
 * databases, tables, views, and other objects to export/backup. It handles
 * the generation of SQL dump statements.
 */
class Backup : public DMdiSubWindow
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the Backup window.
   * @param serverConnection Pointer to the active DBMS connection to be used for retrieval.
   */
  Backup(DBMS *serverConnection);

  /**
   * @brief Retranslates the user interface.
   *
   * Updates all visible text in the widget to reflect the currently selected language.
   */
  void retranslateUI();

signals:
  /**
   * @brief Emitted to update the progress of a loading operation.
   * @param progress The completion percentage (0-100).
   */
  void loadProgress(const int progress);

  /**
   * @brief Emitted to send a status message to the main application.
   * @param message The text message to display.
   * @param timeout The duration in milliseconds to show the message.
   * @param progress A progress value associated with the message.
   */
  void emitMessage(QString message, unsigned int timeout, double progress);

private:
  DBMS *serverConnection;
  DTitleLabel *dTitleLabel;
  QTreeWidget *tablesListWidget;
  QGroupBox *groupBoxAction;
  BaseTextEditor *resultEditor;
  QList<QTreeWidgetItem *> tables;
//  void tableList();
  QCheckBox *addDropStatement;
  QCheckBox *addBackupDate;
  QCheckBox *addBackupTableData;
  DFileSelector *dFileSelector;
  QVBoxLayout *thirdLayout;
  QPushButton *backItUpPushButton;
  QPushButton *selectAllDatabasesPushButton;
  QPushButton *selectAllTablesPushButton;
  QPushButton *selectAllViewsPushButton;
  QPushButton *selectAllEventsPushButton;
  QPushButton *selectAllTriggersPushButton;
  QPushButton *selectAllRoutinesPushButton;
  QPushButton *selectAllUsersPushButton;
  QPushButton *selectAllPushButton;
  QStringList tablesToBackup;
  QProgressDialog *backupProgressDialog;
  bool cancelBackup;

  /**
   * @brief Converts an internal item type identifier to a readable string.
   * @param itemType The integer representing the type (e.g., Table, View).
   * @param object The specific name of the object (optional).
   * @return A QString describing the item type.
   */
  QString textualItemType(int itemType, QString object = QString());

  /**
   * @brief Retrieves the SQL CREATE definition for a specific object.
   * @param itemType The type of the database object.
   * @param object The name of the database object.
   * @return A QString containing the SQL definition statement.
   */
  QString getDefinition(int itemType, QString object);

  /**
   * @brief Counts the total number of objects selected for backup.
   * @return The integer count of selected items.
   */
  int countObjectsToBackup();

private slots:
  /**
   * @brief Populates the tree widget with available databases and their objects.
   */
  void fillDatabasesSlot();

  /**
   * @brief Handles the activation (e.g., click or double-click) of a tree item.
   * @param item The tree item that was activated.
   * @param column The column index where the activation occurred.
   */
  void itemActivatedSlot(QTreeWidgetItem *item, int column);

  /**
   * @brief Executes the backup process.
   *
   * Iterates through selected items, generates SQL statements, and saves the output.
   */
  void backItUpSlot();

//  void readyReadStandardErrorSlot();
//  void finishedSlot(int exitCode);

  /**
   * @brief Handles interactions with the backup progress dialog (e.g., cancellation).
   */
  void backupProgressDialogSlot();

  /**
   * @brief Disables the "Back It Up" button.
   *
   * Used to prevent multiple clicks while an operation is in progress or invalid.
   */
  void disableBackItUp();

  /**
   * @brief Selects all database nodes in the tree widget.
   */
  void selectAllDatabasesPushButtonSlot();

  /**
   * @brief Selects all table nodes in the tree widget.
   */
  void selectAllTablesPushButtonSlot();

  /**
   * @brief Selects all view nodes in the tree widget.
   */
  void selectAllViewsPushButtonSlot();

  /**
   * @brief Selects all event nodes in the tree widget.
   */
  void selectAllEventsPushButtonSlot();

  /**
   * @brief Selects all trigger nodes in the tree widget.
   */
  void selectAllTriggersPushButtonSlot();

  /**
   * @brief Selects all routine (procedures/functions) nodes in the tree widget.
   */
  void selectAllRoutinesPushButtonSlot();

  /**
   * @brief Selects all user nodes in the tree widget.
   */
  void selectAllUsersPushButtonSlot();

  /**
   * @brief Selects all items in the tree widget regardless of type.
   */
  void selectAllPushButtonSlot();
};
#endif // BACKUP_H
