/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#include <QVBoxLayout>
#include <QSplitter>
#include <QTreeWidget>
#include <QGroupBox>
#include <QTimer>
#include <QCheckBox>
#include <QApplication>
#include <QMessageBox>
#include <QPushButton>
#include <QProgressDialog>
#include <QFile>

#include <iostream>
#include <fstream>
using namespace std;

#if defined(Q_OS_WIN)
  #include "mysql.h"
#elif defined(Q_OS_LINUX)
  #include "mysql.h"
#elif defined(Q_OS_MAC)
  #include "mysql.h"
#endif

#include "backup.h"
#include "dbms.h"
#include "dtitlelabel.h"
#include "basetexteditor.h"
#include "dfileselector.h"

#include <QDebug>

Backup::Backup(DBMS *serverConnection)
{
  this->serverConnection = serverConnection;
  setWindowIcon(DIcon::DatabaseBackup());

  QWidget *widMain = new QWidget;
  QVBoxLayout *mainVLayout = new QVBoxLayout;
  mainVLayout->setContentsMargins(3, 0, 3, 0);
  dTitleLabel = new DTitleLabel;
  mainVLayout->addWidget(dTitleLabel);

  tablesListWidget = new QTreeWidget;
  connect(tablesListWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(itemActivatedSlot(QTreeWidgetItem*,int)));

  groupBoxAction = new QGroupBox(this);
  thirdLayout = new QVBoxLayout;

  addDropStatement = new QCheckBox();
  addDropStatement->setAttribute(Qt::WA_AlwaysShowToolTips, true);
  addDropStatement->setChecked(true);
  thirdLayout->addWidget(addDropStatement);

  addBackupDate = new QCheckBox();
  addBackupDate->setAttribute(Qt::WA_AlwaysShowToolTips, true);
  addBackupDate->setChecked(true);
  thirdLayout->addWidget(addBackupDate);

  addBackupTableData = new QCheckBox();
  addBackupTableData->setAttribute(Qt::WA_AlwaysShowToolTips, true);
  addBackupTableData->setChecked(true);
  thirdLayout->addWidget(addBackupTableData);

  QFrame* sepatatorFrame = new QFrame();
  sepatatorFrame->setFrameShape(QFrame::HLine);
  thirdLayout->addWidget(sepatatorFrame);

  selectAllPushButton = new QPushButton;
  selectAllPushButton->setCheckable(true);
  connect(selectAllPushButton, SIGNAL(clicked()), this, SLOT(selectAllPushButtonSlot()));
  thirdLayout->addWidget(selectAllPushButton);
  selectAllDatabasesPushButton = new QPushButton;
  selectAllDatabasesPushButton->setCheckable(true);
  connect(selectAllDatabasesPushButton, SIGNAL(clicked()), this, SLOT(selectAllDatabasesPushButtonSlot()));
  thirdLayout->addWidget(selectAllDatabasesPushButton);
  selectAllTablesPushButton = new QPushButton;
  selectAllTablesPushButton->setCheckable(true);
  connect(selectAllTablesPushButton, SIGNAL(clicked()), this, SLOT(selectAllTablesPushButtonSlot()));
  thirdLayout->addWidget(selectAllTablesPushButton);
  selectAllViewsPushButton = new QPushButton;
  selectAllViewsPushButton->setCheckable(true);
  connect(selectAllViewsPushButton, SIGNAL(clicked()), this, SLOT(selectAllViewsPushButtonSlot()));
  thirdLayout->addWidget(selectAllViewsPushButton);
  selectAllEventsPushButton = new QPushButton;
  selectAllEventsPushButton->setCheckable(true);
  connect(selectAllEventsPushButton, SIGNAL(clicked()), this, SLOT(selectAllEventsPushButtonSlot()));
  thirdLayout->addWidget(selectAllEventsPushButton);
  selectAllTriggersPushButton = new QPushButton;
  selectAllTriggersPushButton->setCheckable(true);
  connect(selectAllTriggersPushButton, SIGNAL(clicked()), this, SLOT(selectAllTriggersPushButtonSlot()));
  thirdLayout->addWidget(selectAllTriggersPushButton);
  selectAllRoutinesPushButton = new QPushButton;
  selectAllRoutinesPushButton->setCheckable(true);
  connect(selectAllRoutinesPushButton, SIGNAL(clicked()), this, SLOT(selectAllRoutinesPushButtonSlot()));
  thirdLayout->addWidget(selectAllRoutinesPushButton);
  selectAllUsersPushButton = new QPushButton;
  selectAllUsersPushButton->setCheckable(true);
  connect(selectAllUsersPushButton, SIGNAL(clicked()), this, SLOT(selectAllUsersPushButtonSlot()));
  thirdLayout->addWidget(selectAllUsersPushButton);

  QFrame* sepatatorFrame2 = new QFrame();
  sepatatorFrame2->setFrameShape(QFrame::HLine);
  thirdLayout->addWidget(sepatatorFrame2);

  thirdLayout->addStretch();
  groupBoxAction->setLayout(thirdLayout);
  QSizePolicy sizePolicy = groupBoxAction->sizePolicy();
  sizePolicy.setHorizontalStretch(0);
  sizePolicy.setVerticalStretch(0);
  groupBoxAction->setSizePolicy(sizePolicy);

  backItUpPushButton = new QPushButton;
  backItUpPushButton->setEnabled(false);
  connect(backItUpPushButton, SIGNAL(clicked()), this, SLOT(backItUpSlot()));
  thirdLayout->addWidget(backItUpPushButton);

  QSplitter *mainSplitter = new QSplitter(Qt::Horizontal);
  QSizePolicy sizePolicy2 = mainSplitter->sizePolicy();
  sizePolicy2.setVerticalPolicy(QSizePolicy::Expanding);
  mainSplitter->setSizePolicy(sizePolicy2);
  mainSplitter->addWidget(tablesListWidget);
  mainSplitter->addWidget(groupBoxAction);

  resultEditor = new BaseTextEditor(EditorTypes::NoEditor);
  resultEditor->setWordWrapMode(QTextOption::WordWrap);
  resultEditor->showHideLineNumbers(false);
  mainSplitter->addWidget(resultEditor);

  dFileSelector = new DFileSelector(DFileSelectorContexts::BackupFile, tr("Backup file"), true, DIcon::DatabaseBackup());
  connect(dFileSelector, SIGNAL(changed()), this, SLOT(disableBackItUp()));
  mainVLayout->addWidget(mainSplitter);
  mainVLayout->addWidget(dFileSelector);
  retranslateUI();
  QTimer::singleShot(0, this, SLOT(fillDatabasesSlot()));
  widMain->setLayout(mainVLayout);
  setWidget(widMain);
}

void Backup::disableBackItUp()
{
  bool hasSelection = false;
  foreach (QTreeWidgetItem *tableItem, tables) {
    if (tableItem->checkState(0)) {
      hasSelection = true;
      break;
    }
  }
  if (!dFileSelector->isEmpty() && hasSelection)
    backItUpPushButton->setEnabled(true);
  else
    backItUpPushButton->setEnabled(false);
}

void Backup::selectAllDatabasesPushButtonSlot()
{
  foreach (QTreeWidgetItem *tableItem, tables)
    if (tableItem->type() == ItemTypes::Database)
      tableItem->setCheckState(0,  selectAllDatabasesPushButton->isChecked() ? Qt::Checked : Qt::Unchecked);
}

void Backup::selectAllTablesPushButtonSlot()
{
  foreach (QTreeWidgetItem *tableItem, tables)
    if (tableItem->type() == ItemTypes::Table)
      tableItem->setCheckState(0,  selectAllTablesPushButton->isChecked() ? Qt::Checked : Qt::Unchecked);
}

void Backup::selectAllViewsPushButtonSlot()
{
  foreach (QTreeWidgetItem *tableItem, tables)
    if (tableItem->type() == ItemTypes::View)
      tableItem->setCheckState(0,  selectAllViewsPushButton->isChecked() ? Qt::Checked : Qt::Unchecked);
}

void Backup::selectAllEventsPushButtonSlot()
{
  foreach (QTreeWidgetItem *tableItem, tables)
    if (tableItem->type() == ItemTypes::Event)
      tableItem->setCheckState(0,  selectAllEventsPushButton->isChecked() ? Qt::Checked : Qt::Unchecked);
}

void Backup::selectAllTriggersPushButtonSlot()
{
  foreach (QTreeWidgetItem *tableItem, tables)
    if (tableItem->type() == ItemTypes::Trigger)
      tableItem->setCheckState(0,  selectAllTriggersPushButton->isChecked() ? Qt::Checked : Qt::Unchecked);
}

void Backup::selectAllRoutinesPushButtonSlot()
{
  foreach (QTreeWidgetItem *tableItem, tables)
    if (tableItem->type() == ItemTypes::Routine)
      tableItem->setCheckState(0,  selectAllRoutinesPushButton->isChecked() ? Qt::Checked : Qt::Unchecked);
}

void Backup::selectAllUsersPushButtonSlot()
{
  foreach (QTreeWidgetItem *tableItem, tables)
    if (tableItem->type() == ItemTypes::User)
      tableItem->setCheckState(0,  selectAllUsersPushButton->isChecked() ? Qt::Checked : Qt::Unchecked);
}

void Backup::selectAllPushButtonSlot()
{
  selectAllDatabasesPushButton->setChecked(selectAllPushButton->isChecked());
  selectAllDatabasesPushButtonSlot();
  selectAllTablesPushButton->setChecked(selectAllPushButton->isChecked());
  selectAllTablesPushButtonSlot();
  selectAllViewsPushButton->setChecked(selectAllPushButton->isChecked());
  selectAllViewsPushButtonSlot();
  selectAllEventsPushButton->setChecked(selectAllPushButton->isChecked());
  selectAllEventsPushButtonSlot();
  selectAllTriggersPushButton->setChecked(selectAllPushButton->isChecked());
  selectAllTriggersPushButtonSlot();
  selectAllRoutinesPushButton->setChecked(selectAllPushButton->isChecked());
  selectAllRoutinesPushButtonSlot();
  selectAllUsersPushButton->setChecked(selectAllPushButton->isChecked());
  selectAllUsersPushButtonSlot();
}

void Backup::retranslateUI()
{
  setWindowTitle(tr("Backup"));
  setObjectName(windowTitle());
  dTitleLabel->setText(tr("Backup"));
  dTitleLabel->setToolTip(dTitleLabel->text());
  tablesListWidget->setWindowTitle(tr("Database objects"));
  tablesListWidget->setToolTip(tablesListWidget->windowTitle());
  tablesListWidget->setHeaderLabel(tablesListWidget->windowTitle());
  groupBoxAction->setTitle(tr("Options"));
  groupBoxAction->setToolTip(groupBoxAction->title());
  dFileSelector->setText(tr("&File:"));
  backItUpPushButton->setText(tr("Create Backup"));
  backItUpPushButton->setToolTip(backItUpPushButton->text());
  selectAllDatabasesPushButton->setText(tr("Select all databases"));
  selectAllDatabasesPushButton->setToolTip(selectAllDatabasesPushButton->text());
  selectAllTablesPushButton->setText(tr("Select all tables"));
  selectAllTablesPushButton->setToolTip(selectAllTablesPushButton->text());
  selectAllViewsPushButton->setText(tr("Select all views"));
  selectAllViewsPushButton->setToolTip(selectAllViewsPushButton->text());
  selectAllEventsPushButton->setText(tr("Select all events"));
  selectAllEventsPushButton->setToolTip(selectAllEventsPushButton->text());
  selectAllTriggersPushButton->setText(tr("Select all triggers"));
  selectAllTriggersPushButton->setToolTip(selectAllTriggersPushButton->text());
  selectAllRoutinesPushButton->setText(tr("Select all routines"));
  selectAllRoutinesPushButton->setToolTip(selectAllRoutinesPushButton->text());
  selectAllUsersPushButton->setText(tr("Select all users"));
  selectAllUsersPushButton->setToolTip(selectAllUsersPushButton->text());
  selectAllPushButton->setText(tr("Select all"));
  selectAllPushButton->setToolTip(selectAllPushButton->text());
  addDropStatement->setText(tr("Add drop statement"));
  addDropStatement->setToolTip(addDropStatement->text());
  addDropStatement->setStatusTip(addDropStatement->toolTip());
  addDropStatement->setWhatsThis(addDropStatement->toolTip());
  addBackupDate->setText(tr("Add backup date"));
  addBackupDate->setToolTip(addBackupDate->text());
  addBackupDate->setStatusTip(addBackupDate->toolTip());
  addBackupDate->setWhatsThis(addBackupDate->toolTip());
  addBackupTableData->setText(tr("Backup table data"));
  addBackupTableData->setToolTip(addBackupTableData->text());
  addBackupTableData->setStatusTip(addBackupTableData->toolTip());
  addBackupTableData->setWhatsThis(addBackupTableData->toolTip());
}

void Backup::fillDatabasesSlot()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  tables.clear();
  QTreeWidgetItem *item;
  QTreeWidgetItem *itemChild;
  QStringList databases = serverConnection->getDatabases(true);
  QStringList databaseObjects;
  int counter = 0;
  emit loadProgress(0);
  foreach (QString database, databases) {
    item = new QTreeWidgetItem((QTreeWidget*)0, QStringList("`" + database + "`"), ItemTypes::Database);
    item->setIcon(0, DIcon::Database());
    item->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    item->setCheckState(0, Qt::Unchecked);
    tables.append(item);
    databaseObjects = serverConnection->database(database)->getLocalTables();
    foreach (QString table, databaseObjects) {
      itemChild = new QTreeWidgetItem(item, QStringList(item->text(0) + ".`" + table + "`"), ItemTypes::Table);
      itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
      itemChild->setIcon(0, DIcon::DatabaseTable());
      itemChild->setCheckState(0, Qt::Unchecked);
      tables.append(itemChild);
    }
    databaseObjects = serverConnection->database(database)->getViews();
    foreach (QString view, databaseObjects) {
      itemChild = new QTreeWidgetItem(item, QStringList(item->text(0) + ".`" + view + "`"), ItemTypes::View);
      itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
      itemChild->setIcon(0, DIcon::DatabaseView());
      itemChild->setCheckState(0, Qt::Unchecked);
      tables.append(itemChild);
    }
    databaseObjects = serverConnection->database(database)->getEvents();
    foreach (QString event, databaseObjects) {
      itemChild = new QTreeWidgetItem(item, QStringList(item->text(0) + ".`" + event + "`"), ItemTypes::Event);
      itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
      itemChild->setIcon(0, DIcon::DatabaseEvent());
      itemChild->setCheckState(0, Qt::Unchecked);
      tables.append(itemChild);
    }
    databaseObjects = serverConnection->database(database)->getTriggers();
    foreach (QString trigger, databaseObjects) {
      itemChild = new QTreeWidgetItem(item, QStringList(item->text(0) + ".`" + trigger + "`"), ItemTypes::Trigger);
      itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
      itemChild->setIcon(0, DIcon::DatabaseTrigger());
      itemChild->setCheckState(0, Qt::Unchecked);
      tables.append(itemChild);
    }
    databaseObjects = serverConnection->database(database)->getRoutines();
    foreach (QString routine, databaseObjects) {
      itemChild = new QTreeWidgetItem(item, QStringList(item->text(0) + ".`" + routine + "`"), ItemTypes::Routine);
      itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
      itemChild->setIcon(0, DIcon::DatabaseRoutine());
      itemChild->setCheckState(0, Qt::Unchecked);
      tables.append(itemChild);
    }
    emit loadProgress((int) (counter * 100 / databases.count()));
  }
  item = new QTreeWidgetItem((QTreeWidget*)0, QStringList(tr("Users")), ItemTypes::UsersTab);
  item->setIcon(0, DIcon::DatabaseUser());
  item->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
  item->setCheckState(0, Qt::Unchecked);
  tables.append(item);
  databaseObjects = serverConnection->getUsers();
  foreach (QString user, databaseObjects) {
    itemChild = new QTreeWidgetItem(item, QStringList(user), ItemTypes::User);
    itemChild->setIcon(0, DIcon::DatabaseUser());
    itemChild->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    itemChild->setCheckState(0, Qt::Unchecked);
    tables.append(itemChild);
  }
  tablesListWidget->clear();
  tablesListWidget->insertTopLevelItems(0, tables);
  tablesListWidget->resizeColumnToContents(0);
  emit loadProgress(100);
  QApplication::restoreOverrideCursor();
}

void Backup::itemActivatedSlot(QTreeWidgetItem *item, int column)
{
//  if (item->type() == ItemTypes::Database) {
    foreach (QTreeWidgetItem *tableItem, tables) {
      if (tableItem->parent() == item)
        tableItem->setCheckState(column, item->checkState(column));
    }
//  }
  disableBackItUp();
}

QString Backup::textualItemType(int itemType, QString object)
{
  switch(itemType) {
  case ItemTypes::Table:
    return "TABLE";
    break;
  case ItemTypes::Database:
    return "DATABASE";
    break;
  case ItemTypes::View:
    return "VIEW";
    break;
  case ItemTypes::Event:
    return "EVENT";
    break;
  case ItemTypes::Trigger:
    return "TRIGGER";
    break;
  case ItemTypes::Routine:
    return serverConnection->routine(object.split(".").at(1), object.split(".").at(0))->routineType();
    break;
  case ItemTypes::User:
    return "USER";
    break;
  }
  return "";
}

QString Backup::getDefinition(int itemType, QString object)
{
  switch(itemType) {
  case ItemTypes::Table:
    return serverConnection->table(object.split(".").at(1), object.split(".").at(0))->getDefinition();
    break;
  case ItemTypes::Database:
    return serverConnection->database(object)->getDefinition();
    break;
  case ItemTypes::View:
    return serverConnection->view(object.split(".").at(1), object.split(".").at(0))->getDefinition();
    break;
  case ItemTypes::Event:
    return serverConnection->databaseEvent(object.split(".").at(1), object.split(".").at(0))->getDefinition();
    break;
  case ItemTypes::Trigger:
    return serverConnection->trigger(object.split(".").at(1), object.split(".").at(0))->getDefinition();
    break;
  case ItemTypes::Routine:
    return serverConnection->routine(object.split(".").at(1), object.split(".").at(0))->getDefinition();
    break;
  case ItemTypes::User:
    return serverConnection->user(object)->getDefinition();
    break;
  }
  return "";
}

int Backup::countObjectsToBackup()
{
  int counter = 0;
  foreach (QTreeWidgetItem *tableItem, tables) {
    if (tableItem->checkState(0)) {
      counter++;
    }
  }
  return counter;
}

void Backup::backItUpSlot()
{
  QApplication::setOverrideCursor(Qt::WaitCursor);
  MYSQL_ROW resultedRow;
  MYSQL_RES *resultedRows;
  MYSQL_FIELD *field;
  bool queryExecutedOk;
  cancelBackup = false;
  resultEditor->clear();
  backupProgressDialog = new QProgressDialog(tr("Backing up data"), tr("Cancel"), 0, countObjectsToBackup() + 1, this);
  connect(backupProgressDialog, SIGNAL(canceled()), this, SLOT(backupProgressDialogSlot()));
  backupProgressDialog->setWindowModality(Qt::WindowModal);
  backupProgressDialog->setAutoClose(false);
  backupProgressDialog->show();
  uint databaseAcumulator = 0;
  uint tableAcumulator = 0;
  uint viewAcumulator = 0;
  uint eventAcumulator = 0;
  uint triggerAcumulator = 0;
  uint routineAcumulator = 0;
  uint userAcumulator = 0;
  uint insertAcumulator = 0;
  std::string statement;
  std::string insertPrefix;
  std::string insertSufix;

  ofstream file(dFileSelector->getFileName().toUtf8());
  QTreeWidgetItemIterator it(tablesListWidget);
  file << "-- -----------------------------------\n";
  file << QString("-- %1\n").arg(tr("Calíope backup")).toUtf8().data();
  file << QString("-- %1: %2\n").arg(tr("Start date and time"), QDateTime::currentDateTime().toString()).toUtf8().data();
  file << "-- -----------------------------------\n\n";
  emit loadProgress(0);
  file << "DELIMITER // -- Start\n";
  file << "SET UNIQUE_CHECKS := 0;\n";
  file << "DELIMITER // -- End\n\n";
  file << "DELIMITER // -- Start\n";
  file << "SET FOREIGN_KEY_CHECKS := 0;\n";
  file << "DELIMITER // -- End\n\n";
  while (*it) {
    if (cancelBackup)
      break;
    if ((*it)->checkState(0) && (*it)->type() != ItemTypes::UsersTab) {
      switch((*it)->type()) {
      case ItemTypes::Table:
        tableAcumulator++;
        break;
      case ItemTypes::Database:
        databaseAcumulator++;
        break;
      case ItemTypes::View:
        viewAcumulator++;
        break;
      case ItemTypes::Event:
        eventAcumulator++;
        break;
      case ItemTypes::Trigger:
        triggerAcumulator++;
        break;
      case ItemTypes::Routine:
        routineAcumulator++;
        break;
      case ItemTypes::User:
        userAcumulator++;
        break;
      }
      resultEditor->insertPlainText(tr("Working on:") + " " + (*it)->text(0) + "\n");
      resultEditor->gotoLine(resultEditor->document()->blockCount(), false);
      backupProgressDialog->setValue(backupProgressDialog->value() + 1);
      emit loadProgress((int) (backupProgressDialog->value() * 100 / backupProgressDialog->maximum()));
      file << "-- -----------------------------------\n";
      if (addBackupDate->isChecked()) {
        file << QString("-- %1. %2: %3\n")
                   .arg((*it)->text(0), tr("Date"), QDateTime::currentDateTime().toString()).toUtf8().data();
      } else {
        file << QString("-- %1\n").arg((*it)->text(0)).toUtf8().data();
      }
      file << "-- -----------------------------------\n";
      if (addDropStatement->isChecked()) {
        file << QString("DELIMITER // -- Start\nDROP %1 IF EXISTS %2;\nDELIMITER // -- End\n\n" )
                .arg(textualItemType((*it)->type(), (*it)->text(0)), (*it)->text(0)).toUtf8().data();
      }
      file << QString("DELIMITER // -- Start\n%1;\nDELIMITER // -- End\n\n")
              .arg(getDefinition((*it)->type(), (*it)->text(0))).toUtf8().data();
      if ((*it)->type() == ItemTypes::Table && addBackupTableData->isChecked()) {
        statement = "SELECT * FROM " + (*it)->text(0).toStdString();
        queryExecutedOk = mysql_real_query(this->serverConnection->getMariadbConnection(), statement.c_str(), statement.size());
        if (!queryExecutedOk)
          emit emitMessage(QString("%1 %2 %3").arg(tr("Error: ")).arg(mysql_errno(this->serverConnection->getMariadbConnection())).arg(mysql_error(this->serverConnection->getMariadbConnection())), 3, 0);
        resultedRows = mysql_store_result(this->serverConnection->getMariadbConnection());
        if (resultedRows) {
          insertPrefix = "INSERT INTO " + (*it)->text(0).toStdString() + " (";
          while((field = mysql_fetch_field(resultedRows))) {
            insertPrefix += "`";
            insertPrefix += field->name;
            insertPrefix += "`, ";
          }
          insertPrefix.erase(insertPrefix.end() -2, insertPrefix.end());
          insertPrefix += ") VALUES (";
          while((resultedRow = mysql_fetch_row(resultedRows))) {
            file << insertPrefix;
            for (int counter = 0; counter < (int) mysql_num_fields(resultedRows); counter++) {
              if (resultedRow[counter] == 0) {
//                qDebug() << strlen(resultedRow[counter]);
                insertSufix += "NULL, ";
              } else {
                // insertSufix += "'";
                // char escapedString[strlen(resultedRow[counter]) * 2];
                // mysql_real_escape_string(this->serverConnection->getMariadbConnection(), escapedString, resultedRow[counter], strlen(resultedRow[counter]));
                // insertSufix += escapedString; //resultedRow[counter];
                // insertSufix += "', ";
                unsigned long sourceLen = strlen(resultedRow[counter]);
                QByteArray escapedBuffer(sourceLen * 2 + 1, 0);
                mysql_real_escape_string(this->serverConnection->getMariadbConnection(),
                                         escapedBuffer.data(),
                                         resultedRow[counter],
                                         sourceLen);
                insertSufix += "'";
                insertSufix += escapedBuffer;
                insertSufix += "', ";
              }
            }
            insertSufix.erase(insertSufix.end() - 2, insertSufix.end());
            file << insertSufix;
            file << ");\n";
            insertSufix = "";
          }
        }
        mysql_free_result(resultedRows);

//        QList<QStringList> *rows2;
//        QString rowTMP;
//        QString rowData;
//        QString formatedColumns;
//        QString fields;
//        QRegularExpression regularExpression("(BINARY|BLOB)", QRegularExpression::CaseInsensitiveOption);
//        rows2 = serverConnection->runQuery("SELECT `COLUMN_NAME`, `DATA_TYPE` FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA` = '" + database + "' AND `TABLE_NAME` = '" + table + "' ORDER BY `ORDINAL_POSITION` ASC");
//        for (int row = 0; row < rows2->count() - 1; row++) {
//          fields += "`" + rows2->at(row).at(0) + "`, ";
//          if (QString(rows2->at(row).at(1)).contains(regularExpression)) {
//            formatedColumns += "HEX(`" + rows2->at(row).at(0) + "`), ";
//          } else {
//            formatedColumns += "`" + rows2->at(row).at(0) + "`, ";
//          }
//        }
//        fields = fields.mid(0, fields.length() - 2);


//        foreach(QString insertStatement, serverConnection->tables()->getTableDataToInsert((*it)->text(0))) {
//          file << QString("%1;\n").arg(insertStatement).toUtf8().data();
//          insertAcumulator++;
//        }
        file << "\n";
      }
      if ((*it)->type() == ItemTypes::User) {
        foreach(QString userGrants, serverConnection->user((*it)->text(0))->getGrants()) {
          file << QString("%1;\n").arg(userGrants).toUtf8().data();
        }
        file << "\n";
      }
    }
    ++it;
  }
  file << "DELIMITER // -- Start\n";
  file << "SET UNIQUE_CHECKS := 1;\n";
  file << "DELIMITER // -- End\n\n";
  file << "DELIMITER // -- Start\n";
  file << "SET FOREIGN_KEY_CHECKS := 1;\n";
  file << "DELIMITER // -- End\n\n";
  file << "-- -----------------------------------\n";
  file << QString("-- %1: %2\n").arg(tr("End date and time"), QDateTime::currentDateTime().toString()).toUtf8().data();
  file << "-- -----------------------------------\n\n";
  file << "-- -----------------------------------\n";
  file << QString("-- %1\n").arg(tr("Object summary")).toUtf8().data();
  file << QString("-- %1: %2\n").arg(tr("Databases"), QString::number(databaseAcumulator)).toUtf8().data();
  file << QString("-- %1: %2\n").arg(tr("Tables"), QString::number(tableAcumulator)).toUtf8().data();
  file << QString("-- %1: %2\n").arg(tr("Views"), QString::number(viewAcumulator)).toUtf8().data();
  file << QString("-- %1: %2\n").arg(tr("Events"), QString::number(eventAcumulator)).toUtf8().data();
  file << QString("-- %1: %2\n").arg(tr("Triggers"), QString::number(triggerAcumulator)).toUtf8().data();
  file << QString("-- %1: %2\n").arg(tr("Routines"), QString::number(routineAcumulator)).toUtf8().data();
  file << QString("-- %1: %2\n").arg(tr("Users"), QString::number(userAcumulator)).toUtf8().data();
  file << QString("-- %1: %2\n").arg(tr("Inserts"), QString::number(insertAcumulator)).toUtf8().data();
  file << "-- -----------------------------------\n\n";
  file.close();

//  QFile file(dFileSelector->getFileName());
//  file.open(QFile::WriteOnly | QFile::Text);
//  QTreeWidgetItemIterator it(tablesListWidget);
//  file.write("-- -----------------------------------\n");
//  file.write("-- " + tr("Calíope backup").toLocal8Bit() + "\n");
//  file.write(QString("-- %1: %2\n").arg(tr("Start date and time"), QDateTime::currentDateTime().toString()).toLocal8Bit());
//  file.write("-- -----------------------------------\n\n");
//  emit loadProgress(0);
//  file.write("DELIMITER // -- Start\n");
//  file.write("SET UNIQUE_CHECKS := 0;\n");
//  file.write("SET FOREIGN_KEY_CHECKS := 0;\n");
//  file.write("DELIMITER // -- End\n");
//  while (*it) {
//    if (cancelBackup)
//      break;
//    if ((*it)->checkState(0)) {
//      switch((*it)->type()) {
//      case ItemTypes::Table:
//        tableAcumulator++;
//        break;
//      case ItemTypes::Database:
//        databaseAcumulator++;
//        break;
//      case ItemTypes::View:
//        viewAcumulator++;
//        break;
//      case ItemTypes::Event:
//        eventAcumulator++;
//        break;
//      case ItemTypes::Trigger:
//        triggerAcumulator++;
//        break;
//      case ItemTypes::Routine:
//        routineAcumulator++;
//        break;
//      case ItemTypes::User:
//        userAcumulator++;
//        break;
//      }
//      resultEditor->insertPlainText(tr("Working on:") + " " + (*it)->text(0) + "\n");
//      resultEditor->gotoLine(resultEditor->document()->blockCount(), false);
//      backupProgressDialog->setValue(backupProgressDialog->value() + 1);
//      emit loadProgress((int) (backupProgressDialog->value() * 100 / backupProgressDialog->maximum()));
//      file.write("-- -----------------------------------\n");
//      if (addBackupDate->isChecked()) {
//        file.write(QString("-- %1. %2: %3\n")
//                           .arg((*it)->text(0).toLocal8Bit(), tr("Date"), QDateTime::currentDateTime().toString()).toLocal8Bit());
//      } else {
//        file.write("-- " + (*it)->text(0).toLocal8Bit() + "\n");
//      }
//      file.write("-- -----------------------------------\n");
//      if (addDropStatement->isChecked()) {
//        file.write(QString("DELIMITER // -- Start\nDROP %1 IF EXISTS %2;\nDELIMITER // -- End\n\n" ).arg(textualItemType((*it)->type(), (*it)->text(0)), (*it)->text(0)).toLocal8Bit());
//      }
//      file.write(QString("DELIMITER // -- Start\n%1;\nDELIMITER // -- End\n\n").arg(getDefinition((*it)->type(), (*it)->text(0))).toLocal8Bit());
//      if ((*it)->type() == ItemTypes::Table && addBackupTableData->isChecked()) {
//        foreach(QString insertStatement, serverConnection->tables()->getTableDataToInsert((*it)->text(0))) {
//          file.write(QString("%1;\n").arg(insertStatement).toLocal8Bit());
//          insertAcumulator++;
//        }
//        file.write("\n");
//      }
//      if ((*it)->type() == ItemTypes::User) {
//        foreach(QString userGrants, serverConnection->user((*it)->text(0))->getGrants()) {
//          file.write(QString("%1;\n").arg(userGrants).toLocal8Bit());
//        }
//        file.write("\n");
//      }
//    }
//    ++it;
//  }
//  file.write("DELIMITER // -- Start\n");
//  file.write("SET UNIQUE_CHECKS := 1;\n");
//  file.write("SET FOREIGN_KEY_CHECKS := 1;\n");
//  file.write("DELIMITER // -- End\n");
//  file.write("-- -----------------------------------\n");
//  file.write(QString("-- %1: %2\n").arg(tr("End date and time"), QDateTime::currentDateTime().toString()).toLocal8Bit());
//  file.write("-- -----------------------------------\n\n");
//  file.write("-- -----------------------------------\n");
//  file.write(QString("-- %1\n").arg(tr("Object summary")).toLocal8Bit());
//  file.write(QString("-- %1: %2\n").arg(tr("Databases"), QString::number(databaseAcumulator)).toLocal8Bit());
//  file.write(QString("-- %1: %2\n").arg(tr("Tables"), QString::number(tableAcumulator)).toLocal8Bit());
//  file.write(QString("-- %1: %2\n").arg(tr("Views"), QString::number(viewAcumulator)).toLocal8Bit());
//  file.write(QString("-- %1: %2\n").arg(tr("Events"), QString::number(eventAcumulator)).toLocal8Bit());
//  file.write(QString("-- %1: %2\n").arg(tr("Triggers"), QString::number(triggerAcumulator)).toLocal8Bit());
//  file.write(QString("-- %1: %2\n").arg(tr("Routines"), QString::number(routineAcumulator)).toLocal8Bit());
//  file.write(QString("-- %1: %2\n").arg(tr("Users"), QString::number(userAcumulator)).toLocal8Bit());
//  file.write(QString("-- %1: %2\n").arg(tr("Inserts"), QString::number(insertAcumulator)).toLocal8Bit());
//  file.write("-- -----------------------------------\n\n");
//  file.close();
  emit loadProgress(100);
  QApplication::restoreOverrideCursor();
  QString message(!cancelBackup ? tr("Backup done") : tr("Backup canceled"));
  backupProgressDialog->close();
  QMessageBox::information(this, message, message);
}

void Backup::backupProgressDialogSlot()
{
  cancelBackup = true;
}
