/*****************************************************************************
*
* This file is part of Calíope.
* Copyright (c) 2008-2026 David Villalobos Cambronero (david.villalobos.c@gmail.com).
*
* Calíope is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Calíope is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
*
*****************************************************************************/

#ifndef CALIOPEUPDATER_H
#define CALIOPEUPDATER_H

#include <QWidget>
#include <QNetworkAccessManager>
#include <QStringList>

class QNetworkReply;
class QDialogButtonBox;
class QLabel;

/**
 * @brief The CaliopeUpdater class handles the application auto-update process.
 *
 * This widget is responsible for checking a remote server for new versions of the
 * software, downloading the updated executables or packages, and initiating the
 * installation or restart process. It uses QNetworkAccessManager for HTTP communication.
 */
class CaliopeUpdater : public QWidget
{
  Q_OBJECT

public:
  /**
   * @brief Constructs the CaliopeUpdater widget.
   *
   * Initializes the network managers and checks the current OS environment
   * to determine the correct update path.
   */
  CaliopeUpdater();

  /**
   * @brief Destructor.
   */
  ~CaliopeUpdater();

private slots:
  /**
   * @brief Handles the response from the initial version check request.
   *
   * Parses the reply to determine if a newer version is available compared
   * to the local version.
   *
   * @param reply The network reply containing version information.
   */
  void downloadFinished(QNetworkReply *reply);

  /**
   * @brief Handles the completion of the update file download.
   *
   * Saves the downloaded binary or package to the appropriate temporary location
   * and prepares for installation.
   *
   * @param reply The network reply containing the file data.
   */
  void downloadFinished2(QNetworkReply *reply);

  /**
   * @brief Slot triggered when the user confirms the update (OK button).
   *
   * Starts the actual download or installation process.
   */
  void okSlot();

  /**
   * @brief Slot triggered when the user cancels the update (Cancel button).
   *
   * Aborts any ongoing network requests and closes the widget.
   */
  void cancelSlot();

  /**
   * @brief Closes the updater widget.
   */
  void closeSlot();

private:
  /**
   * @brief Retrieves the current version number of the application.
   * @return A string representing the version (e.g., "1.0.2").
   */
  QString getVersionNumber();

  QNetworkAccessManager manager;
  QNetworkAccessManager manager2;
  QStringList listOfExecutables;
  bool fileDownLoadFinished;
  bool fileDownLoadFinished2;
  QString os;
  QDialogButtonBox *buttonBox;
  QString executable;
  QString applicationDirPath;
  QLabel *label;
};
#endif // CALIOPEUPDATER_H
